import os
import subprocess
from pathlib import Path
from custom_logger import MeetingLogger
meeting_logger = MeetingLogger()


def join_wav_files(meeting_path:Path, meeting_id):
    """Join .wav files into a single file."""
    meeting_logger.log_info(meeting_id, f"Joining .wav files in {meeting_path}")
    # 🔴 Remove existing output.wav if it exists
    output_wav = meeting_path / "output.wav"
    if output_wav.exists():
        meeting_logger.log_info(meeting_id, f"Removing existing output file: {output_wav}")
        output_wav.unlink()
    wav_files = sorted(str(p) for p in meeting_path.glob("*.wav"))
    if not wav_files:
        meeting_logger.log_error(meeting_id, "No .wav files found in the meeting directory.")
        raise Exception("No .wav files found.")

    concat_file_path = meeting_path / "concat_list.txt"
    with open(concat_file_path, "w", encoding="utf-8") as f:
        for wav_file in wav_files:
            f.write(f"file '{wav_file}'\n")

    output_wav = os.path.join(meeting_path, "output.wav")
    cmd = [
        "ffmpeg", "-y",
        "-f", "concat", "-safe", "0",
        "-i", str(concat_file_path),
        "-c", "copy",
        output_wav
    ]

    subprocess.run(cmd, check=True)
    meeting_logger.log_info(meeting_id, f"Output WAV file created at: {output_wav}")

    # Optional: Cleanup concat list
    os.remove(concat_file_path)

    return output_wav


if __name__ == "__main__":
    # Example usage
    meeting_id = "Test"
    meeting_path = Path(r"C:\Users\ssenp\Downloads")
    meeting_path = meeting_path / meeting_id
    try:
        output_file = join_wav_files(meeting_path, meeting_id)
        print(f"Joined WAV file created at: {output_file}")
    except Exception as e:
        meeting_logger.log_error(meeting_id, str(e))
        print(f"Error: {str(e)}")