# app/routers/github_router.py
from fastapi import APIRouter, Request, HTTPException, Query, Depends, BackgroundTasks, status
from fastapi.responses import RedirectResponse, JSONResponse
from sqlalchemy.ext.asyncio import AsyncSession
import json
import uuid
from datetime import datetime, timedelta

from app.db.postgress_db import get_async_db
from app.services.postgress_db_service import pg_db_service as database_service
from app.core.config import settings
from app.core.logger import logger
from app.services.github.github_client import github_clinet
from app.schemas.github_schemas import (
    # Request schemas
    GitHubInstallRequest,
    TestWebhookRequest,
    ManualSyncRequest,
    
    # Response schemas
    RepositoryListConfigResponse,
    GitHubCallbackResponse,
    BranchListConfResponse,
    WebhookTestResponse,
    ManualSyncResponse,
    BranchInfo,
    RepositoryInfo,

)
from app.schemas.error_schemas import ErrorResponse

from app.services.webhook.background_processor import webhook_processor
from app.services.github.github_client import GitHubClient


# Constants
INSTALLATION_WAIT_TIMEOUT = settings.INSTALLATION_WAIT_TIMEOUT
WEBHOOK_PROCESSING_DELAY = settings.WEBHOOK_PROCESSING_DELAY
MAX_WEBHOOK_PAYLOAD_SIZE = settings.MAX_WEBHOOK_PAYLOAD_SIZE

github_router = APIRouter()

@github_router.post(
    "/install",
    summary="Install GitHub App",
    response_model=GitHubCallbackResponse,
    responses={
        400: {"model": ErrorResponse},
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def github_app_install(
    request: GitHubInstallRequest,
    db: AsyncSession = Depends(get_async_db)
):
    """Redirect users to install the GitHub App with proper user validation."""
    try:
        user_id = request.user_id
        # Generate installation URL
        install_url = await github_clinet.get_new_installation_user(user_id=user_id)
        
        logger.info(f"Generated installation URL for user: {user_id}")
        
        return GitHubCallbackResponse(
            message="GitHub App installation initiated",
            setup_action="install",
            install_url=install_url
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in GitHub install for user {request.user_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error during GitHub App installation"
        )
    
@github_router.get(
    "/callback",
    summary="GitHub OAuth Callback",
    response_model=GitHubCallbackResponse,
    responses={
        400: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def github_oauth_callback(
    code: str = Query(..., description="OAuth code from GitHub"),
    state: str = Query(None, description="Internal user ID passed during OAuth"),
    setup_action: str = Query(None, description="Setup action from GitHub"),
    installation_id: int = Query(None, description="GitHub App installation ID"),
    db: AsyncSession = Depends(get_async_db)
):
    """Handle GitHub OAuth callback to link user account with GitHub data."""
    try:
        if not state:
            #redirect to Git itself
            return RedirectResponse(url=settings.APP_DOMAIN)
        # Validate parameters
        if not code :
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Missing required parameters: code"
            )
        github_client = GitHubClient()
        # Exchange code for access token
        access_token = await github_client.exchange_code_for_token(code)
        if not access_token:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Failed to exchange code for access token"
            )
        
        logger.info(f"Exchanged code for access token successfully for access_token: {access_token}")
        
        # Get user info from GitHub using access token
        github_user_data = await github_client.get_user_info(access_token)
        if not github_user_data:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Failed to retrieve user data from GitHub"
            )
        
        github_id = github_user_data.get("id")
        username = github_user_data.get("login")
        email = github_user_data.get("email")
        account_type = github_user_data.get("type", "User")
        
        if not github_id or not username:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Invalid GitHub user data received"
            )
        
        # If email is not available from GitHub, create a placeholder
        if not email:
            email = f"{username}@github.user"
        
        # Find user by state (user ID) or create new user
        try:
            user_id = int(state)
            user = await database_service.get_user_by_id(db, user_id)
            
            if not user:
                # Create new user since state user ID doesn't exist
                user_data = {
                    "id": user_id,
                    "email": email,
                    "github_username": username,
                    "github_id": github_id,
                    "github_login_code": code,
                    "is_active": True
                }
                user = await database_service.create_user(db, user_data)
                logger.info(f"Created new user with ID: {user_id}")
            # else:
            #     # User exists, update GitHub info if needed
            #     update_data = {}
            #     if user.github_username != username:
            #         update_data["github_username"] = username
            #     if user.github_login_code != code:
            #         update_data["github_login_code"] = code
                
            #     if update_data:
            #         user = await database_service.update_user(db, user_id, update_data)
            #         logger.info(f"Updated user GitHub info for user ID: {user_id}")
                    
        except ValueError:
            # If state is not a valid integer, create new user with auto-generated ID
            user_data = {
                "email": email,
                "github_username": username,
                "github_id": github_id,
                "github_login_code": code,
                "is_active": True
            }
            user = await database_service.create_user(db, user_data)
            logger.info(f"Created new user with auto-generated ID: {user.id}")
        if installation_id:
            # Create or update GitHub account record
            
            existing_account = await database_service.get_github_account_by_installation_id(db, installation_id)
            if existing_account:
                github_account_data = {
                    "user_id": user.id,
                    "is_active": True,
                }
                # Update existing account
                await database_service.update_github_account(
                    db, existing_account.id, github_account_data
                )
                logger.info(f"Updated GitHub account for installation ID: {installation_id}")
            else:
                # Create new GitHub account
                github_account_data = {
                    "user_id": user.id,
                    "github_username": username,
                    "github_id": github_id,
                    "installation_id": installation_id,
                    "account_type": account_type,
                    "avatar_url": github_user_data.get("avatar_url"),
                    "html_url": github_user_data.get("html_url"),
                    "permissions": {},  # Permissions can be fetched later
                    "is_active": True,
                    "created_at": datetime.utcnow()
                }
                await database_service.create_github_account(db, github_account_data)
                logger.info(f"Created GitHub account for installation ID: {installation_id}")
            return GitHubCallbackResponse(
                message="GitHub account linked successfully",
                setup_action=setup_action,

            )
        install_url= await github_clinet.get_new_installation_user(user_id=user.id)
        return GitHubCallbackResponse( 
            message="GitHub OAuth completed successfully but ask administration to install the app on the selected organization by using the install_url",
            setup_action=setup_action,
            install_url=install_url,
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in GitHub OAuth callback: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error during GitHub OAuth callback"
        )
@github_router.get(
    "/{user_id}/repositories",
    response_model=RepositoryListConfigResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user_repositories(
    user_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """Fetch all repositories for a user."""
    try:
        # Get user's GitHub accounts
        github_accounts = await database_service.get_github_accounts_by_user_id(db, user_id)
        if not github_accounts:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="No GitHub accounts found for user"
            )
        
        # Get all repositories for user's GitHub accounts
        all_repositories = []
        for account in github_accounts:
            if account.is_active:
                repos = await database_service.get_repositories_by_account(db, account.id)
                all_repositories.extend(repos)
        
        # Format response
        formatted_repos = []
        for repo in all_repositories:
            formatted_repos.append(RepositoryInfo(
                id=repo.repo_id,
                name=repo.name,
                full_name=repo.full_name,
                private=repo.private,
                html_url=repo.html_url,
                description=repo.description,
                language=repo.language,
                updated_at=repo.updated_at or repo.created_at,
                visibility="private" if repo.private else "public",
                default_branch=repo.default_branch,
                archived=getattr(repo, 'archived', False)
            ))
        
        private_count = len([r for r in all_repositories if r.private])
        public_count = len(all_repositories) - private_count
        
        return RepositoryListConfigResponse(
            user_id=user_id,
            total_repositories=len(all_repositories),
            private_repositories=private_count,
            public_repositories=public_count,
            repositories=formatted_repos
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error fetching repositories for user {user_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error fetching repositories"
        )

@github_router.get(
    "/repositories/{repository_id}/branches",
    response_model=BranchListConfResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_repository_branches(
    repository_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """Get all branches for a specific repository."""
    try:
        # Get repository
        repository = await database_service.get_repository_by_archea_repo_id(db, repository_id)
        if not repository:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="Repository not found"
            )
        
        # Get branches
        branches = await database_service.get_branches_by_repository(db, repository.id)
        
        # Format response
        formatted_branches = []
        for branch in branches:
            formatted_branches.append(BranchInfo(
                id=branch.id,
                name=branch.name,
                commit_sha=branch.commit_sha,
                commit_message=branch.commit_message,
                commit_author=branch.commit_author,
                is_protected=branch.is_protected,
                last_synced_at=branch.last_synced_at
            ))
        
        return BranchListConfResponse(
            repository_id=repository.id,
            repository_name=repository.name,
            total_branches=len(branches),
            branches=formatted_branches
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting branches for repository {repository_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error fetching branches"
        )
@github_router.post(
    "/webhook",
    summary="GitHub Webhook Handler",
    responses={
        400: {"model": ErrorResponse},
        403: {"model": ErrorResponse},
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def github_webhook(
    request: Request,
    background_tasks: BackgroundTasks,
    db: AsyncSession = Depends(get_async_db)
):
    """Receive GitHub App webhook events with production-grade validation."""
    archea_webhook_id = None
    delivery_id = request.headers.get("X-GitHub-Delivery", str(uuid.uuid4()))
    
    try:
        # Read and verify webhook payload
        body_bytes = await request.body()
        
        # Check payload size
        if len(body_bytes) > MAX_WEBHOOK_PAYLOAD_SIZE:
            raise HTTPException(
                status_code=status.HTTP_413_CONTENT_TOO_LARGE,
                detail=f"Webhook payload too large. Maximum size is {MAX_WEBHOOK_PAYLOAD_SIZE} bytes"
            )
        
        signature = request.headers.get("X-Hub-Signature-256", "")
        event_type = request.headers.get("X-GitHub-Event", "unknown")
        
        # Verify webhook signature
        if not signature:
            logger.warning(f"Missing signature for webhook delivery: {delivery_id}")
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Missing webhook signature"
            )
        
        # Verify signature using GitHubClient
        github_client = GitHubClient()
        if not github_client.verify_webhook_signature(body_bytes, signature):
            logger.warning(f"Invalid webhook signature for delivery: {delivery_id}")
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail="Invalid webhook signature"
            )
        
        # Parse payload
        try:
            payload = json.loads(body_bytes.decode('utf-8'))
        except json.JSONDecodeError as e:
            logger.error(f"Invalid JSON in webhook payload for {delivery_id}: {str(e)}")
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Invalid JSON payload"
            )
        
        # Store webhook in database with minimal processing
        webhook_data = {
            "github_webhook_id": int(request.headers.get("X-GitHub-Hook-ID", 0)),
            "event_type": event_type,
            "payload": payload,
            "headers": dict(request.headers),
            "delivery_id": delivery_id,
            "signature": signature[:255],  # Truncate if too long
            "status": "received",
            "created_at": datetime.utcnow()
        }
        
        # Create webhook record
        webhook = await database_service.create_webhook(db, webhook_data)
        archea_webhook_id = webhook.id

        logger.info(
            f"Webhook received: {event_type} "
            f"(delivery: {delivery_id}, archea_webhook_id: {archea_webhook_id})"
        )
        
        # ✅ MOVE ALL PROCESSING TO BACKGROUND TASK
        background_tasks.add_task(
            webhook_processor.process_webhook,
            archea_webhook_id=archea_webhook_id,
            db_session_context=get_async_db
        )
        
        return JSONResponse(
            status_code=status.HTTP_202_ACCEPTED,
            content={
                "status": "accepted",
                "event": event_type,
                "webhook_id": archea_webhook_id,
                "delivery_id": delivery_id,
                "message": "Webhook is being processed in background"
            }
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Unexpected error processing webhook {delivery_id}: {str(e)}")
        import sys
        import os
        exc_type, exc_obj, exc_tb = sys.exc_info()
        fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
        logger.error(f"Exception type: {exc_type}, File name: {fname}, Line number: {exc_tb.tb_lineno}")
        
        # Update webhook status if it was created
        if archea_webhook_id:
            try:
                await database_service.update_webhook_status(
                    db, archea_webhook_id, "failed", str(e)
                )
            except Exception as update_error:
                logger.error(f"Failed to update webhook status: {str(update_error)}")
        
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error processing webhook"
        )

@github_router.post(
    "/webhook/test",
    response_model=WebhookTestResponse,
    responses={500: {"model": ErrorResponse}}
)
async def test_webhook(
    request: TestWebhookRequest,
    background_tasks: BackgroundTasks,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Test webhook endpoint - accepts event and payload in request body.
    Uses the same background processor as production webhooks.
    """
    try:
        event = request.event
        payload = request.payload
        
        # Basic validation
        installation_data = payload.get("installation", {})
        installation_id = installation_data.get("id")
        
        if not installation_id:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Installation ID required in payload"
            )
        
        # Store test webhook in database (minimal processing - same as production)
        webhook_data = {
            "github_webhook_id": 0,  # Test webhook
            "event_type": event,
            "payload": payload,
            "headers": {"test": "true"},
            "delivery_id": f"test-{uuid.uuid4()}",
            "signature": "test-signature",
            "status": "received",
            "installation_id": installation_id,
            "created_at": datetime.utcnow()
        }
        
        # Create webhook record
        webhook = await database_service.create_webhook(db, webhook_data)
        archea_webhook_id = webhook.id
        
        logger.info(f"TEST WEBHOOK - Created test webhook record (ID: {archea_webhook_id}) for event: {event}")

        # ✅ USE THE SAME BACKGROUND PROCESSOR AS PRODUCTION
        background_tasks.add_task(
            webhook_processor.process_webhook,
            archea_webhook_id=archea_webhook_id,
            db_session_context=get_async_db
        )
        
        actions_performed = [
            f"✓ Created test webhook record (ID: {archea_webhook_id})",
            f"✓ Queued for background processing via WebhookBackgroundProcessor",
            f"✓ Event type: {event}",
            f"✓ Installation ID: {installation_id}"
        ]

        return WebhookTestResponse(
            status="test_executed",
            event=event,
            webhook_id=archea_webhook_id,
            message="Webhook test executed - processing in background (same as production)",
            actions_performed=actions_performed,
            notes=[
                "✓ Webhook recorded in database",
                "✓ Using same background processor as production",
                "✓ GitHub account will be created/updated in background",
                "✓ User association will be handled in background",
                "✓ Actual GitHub operations will be performed",
                "✓ Data will be stored in PostgreSQL",
                "✓ Check webhook status via /webhooks endpoint to see progress"
            ]
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in test webhook: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error executing test webhook: {str(e)}"
        )
@github_router.post(
    "/sync",
    response_model=ManualSyncResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def manual_sync_repositories(
    request: ManualSyncRequest,
    background_tasks: BackgroundTasks,
    db: AsyncSession = Depends(get_async_db)
):
    """Manually trigger repository sync for a user using webhook pipeline."""
    try:
        user_id = request.user_id
        
        # Get user's active GitHub accounts
        github_accounts = await database_service.get_github_accounts_by_user_id(db, user_id)
        active_accounts = [acc for acc in github_accounts if acc.is_active]
        
        if not active_accounts:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="No active GitHub installations found for user"
            )
        
        # Use the first active account
        account = active_accounts[0]
        
        # Create a synthetic webhook for manual sync
        webhook_data = {
            "github_webhook_id": 0,  # Manual sync
            "event_type": "installation",  # Use installation event to trigger full sync
            "payload": {
                "action": "sync_manual",
                "installation": {
                    "id": account.installation_id,
                    "account": {
                        "login": account.github_username,
                        "id": account.github_id
                    }
                }
            },
            "headers": {"manual-sync": "true"},
            "delivery_id": f"manual-sync-{uuid.uuid4()}",
            "signature": "manual-sync-signature",
            "user_id": user_id,
            "installation_id": account.installation_id,
            "status": "received",
            "created_at": datetime.utcnow()
        }
        
        # Create webhook record
        webhook = await database_service.create_webhook(db, webhook_data)
        archea_webhook_id = webhook.id
        
        logger.info(f"Manual sync initiated for user {user_id}, webhook {archea_webhook_id}")
        
        # ✅ USE THE SAME BACKGROUND PROCESSOR AS WEBHOOKS
        background_tasks.add_task(
            webhook_processor.process_webhook,
            archea_webhook_id=archea_webhook_id,
            db_session_context=get_async_db
        )
        
        return ManualSyncResponse(
            message=f"Manual sync initiated for user {user_id}",
            user_id=user_id,
            installation_id=account.installation_id,
            webhook_id=archea_webhook_id,
            timestamp=datetime.utcnow(),
            notes=[
                "✓ Using same processing pipeline as webhooks",
                "✓ Full repository sync will be performed",
                "✓ Check webhook status via /webhooks endpoint",
                "✓ All repositories, branches, and files will be synced"
            ]
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in manual sync for user {request.user_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error initiating manual sync: {str(e)}"
        )