# app/routers/user_router.py
from fastapi import APIRouter, Depends, HTTPException, status, Query
from sqlalchemy.ext.asyncio import AsyncSession
from typing import List, Optional, Dict, Any
import json
from app.db.postgress_db import get_async_db
from app.services.postgress_db_service import pg_db_service as database_service
from app.models.user_models import User, UserGitHubAccount
from app.models.github_models import UserRepository, UserBranch, Webhook
from app.models.sync_models import SyncJob, CodeChange
from sqlalchemy import select, func
from sqlalchemy.orm import selectinload
from app.core.logger import logger

# Import schemas
from app.schemas import (
    # User schemas
     UserResponse, UserListResponse,
    # GitHub schemas
    GitHubAccountCreate, GitHubAccountResponse, GitHubAccountListResponse, 
    GitHubAccountCreateResponse, RepositoryResponse, RepositoryListResponse,
    
    # Sync schemas
    SyncJobSummary, UserSyncStatusResponse, SyncStatistics,
    ActivitySummary, UserActivityResponse,
    
    # Health schemas
    AccountHealth, UserHealthResponse,
    
    # Error schemas
    ErrorResponse, HTTPValidationError,
    
    # Common schemas
    PaginationParams, ActivityPagination,

    # Webhook 
    WebhookListResponse,
    WebhookResponse
)

user_router = APIRouter()

# ===== USER MANAGEMENT =====

@user_router.get(
    "/{user_id}", 
    response_model=UserResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user(
    user_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Get user details by ID
    """
    logger.info(f"Fetching user details for user_id: {user_id}")
    try:
        result = await db.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            logger.warning(f"User not found - user_id: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )
        
        logger.debug(f"Successfully retrieved user: {user_id}")
        return UserResponse.from_orm(user)
        
    except HTTPException:
        logger.warning(f"HTTP exception while fetching user: {user_id}")
        raise
    except Exception as e:
        logger.error(f"Error retrieving user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error retrieving user: {str(e)}"
        )

@user_router.get(
    "/", 
    response_model=UserListResponse,
    responses={500: {"model": ErrorResponse}}
)
async def list_users(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    db: AsyncSession = Depends(get_async_db)
):
    """
    List all users with pagination
    """
    logger.info(f"Listing users - skip: {skip}, limit: {limit}")
    try:
        # Get total count
        count_result = await db.execute(select(func.count(User.id)))
        total_count = count_result.scalar()
        logger.debug(f"Total users count: {total_count}")
        
        # Get users
        result = await db.execute(
            select(User)
            .order_by(User.created_at.desc())
            .offset(skip)
            .limit(limit)
        )
        users = result.scalars().all()
        
        user_list = [UserResponse.from_orm(user) for user in users]
        logger.debug(f"Retrieved {len(user_list)} users")
        
        return UserListResponse(
            total_count=total_count,
            skip=skip,
            limit=limit,
            users=user_list
        )
        
    except Exception as e:
        logger.error(f"Error listing users (skip: {skip}, limit: {limit}): {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error listing users: {str(e)}"
        )

# ===== GITHUB ACCOUNT MANAGEMENT =====

# @user_router.post(
#     "/{user_id}/github-accounts", 
#     response_model=GitHubAccountCreateResponse,
#     responses={
#         400: {"model": ErrorResponse},
#         404: {"model": ErrorResponse},
#         500: {"model": ErrorResponse}
#     }
# )
async def add_github_account(
    user_id: int,
    github_data: GitHubAccountCreate,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Add a GitHub account to a user
    """
    logger.info(f"Adding GitHub account for user_id: {user_id}, installation_id: {github_data.installation_id}")
    try:
        # Check if user exists
        logger.debug(f"Verifying user existence - user_id: {user_id}")
        result = await db.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            logger.warning(f"User not found while adding GitHub account - user_id: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )

        # Check if GitHub account already exists
        logger.debug(f"Checking if GitHub account already exists - installation_id: {github_data.installation_id}")
        existing_account = await database_service.get_github_account_by_installation_id(
            db, github_data.installation_id
        )
        if existing_account:
            logger.warning(f"GitHub account already exists - installation_id: {github_data.installation_id}")
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="GitHub account with this installation ID already exists"
            )

        # Create GitHub account
        account_data = github_data.dict()
        account_data["user_id"] = user_id
        logger.debug(f"Creating GitHub account with data: {account_data}")
        github_account = await database_service.create_github_account(db, account_data)
        
        logger.info(f"GitHub account added successfully - account_id: {github_account.id}, username: {github_account.github_username}")
        return GitHubAccountCreateResponse(
            message="GitHub account added successfully",
            account_id=github_account.id,
            github_username=github_account.github_username,
            installation_id=github_account.installation_id
        )
        
    except HTTPException:
        logger.warning(f"HTTP exception while adding GitHub account for user {user_id}")
        raise
    except Exception as e:
        logger.error(f"Error adding GitHub account for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error adding GitHub account: {str(e)}"
        )

@user_router.get(
    "/{user_id}/github-accounts", 
    response_model=GitHubAccountListResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user_github_accounts(
    user_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Get all GitHub accounts for a user
    """
    logger.info(f"Fetching GitHub accounts for user_id: {user_id}")
    try:
        # Check if user exists
        logger.debug(f"Verifying user existence - user_id: {user_id}")
        result = await db.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            logger.warning(f"User not found while fetching GitHub accounts - user_id: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )

        # Get GitHub accounts
        logger.debug(f"Querying GitHub accounts for user_id: {user_id}")
        result = await db.execute(
            select(UserGitHubAccount)
            .where(UserGitHubAccount.user_id == user_id)
            .order_by(UserGitHubAccount.created_at.desc())
        )
        accounts = result.scalars().all()
        
        account_list = [GitHubAccountResponse.from_orm(account) for account in accounts]
        logger.debug(f"Retrieved {len(account_list)} GitHub accounts for user {user_id}")
        
        return GitHubAccountListResponse(
            user_id=user_id,
            total_accounts=len(account_list),
            github_accounts=account_list
        )
        
    except HTTPException:
        logger.warning(f"HTTP exception while fetching GitHub accounts for user {user_id}")
        raise
    except Exception as e:
        logger.error(f"Error retrieving GitHub accounts for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error retrieving GitHub accounts: {str(e)}"
        )

# ===== USER REPOSITORIES =====

@user_router.get(
    "/{user_id}/repositories", 
    response_model=RepositoryListResponse,
    responses={500: {"model": ErrorResponse}}
)
async def get_user_repositories(
    user_id: int,
    github_username: Optional[str] = Query(None),
    db: AsyncSession = Depends(get_async_db)
):
    """
    Get all repositories for a user, optionally filtered by GitHub account
    """
    logger.info(f"Fetching repositories for user_id: {user_id}, github_username: {github_username}")
    try:
        # Build query
        query = (
            select(UserRepository)
            .join(UserGitHubAccount)
            .where(UserGitHubAccount.user_id == user_id)
        )
        
        if github_username:
            query = query.where(UserGitHubAccount.github_username == github_username)
            logger.debug(f"Filtering repositories by GitHub username: {github_username}")
        
        query = query.order_by(UserRepository.created_at.desc())
        
        result = await db.execute(query)
        repositories = result.scalars().all()
        
        repo_list = [RepositoryResponse.from_orm(repo) for repo in repositories]
        logger.debug(f"Retrieved {len(repo_list)} repositories for user {user_id}")
        
        return RepositoryListResponse(
            user_id=user_id,
            github_username=github_username,
            total_repositories=len(repo_list),
            repositories=repo_list
        )
        
    except Exception as e:
        logger.error(f"Error retrieving repositories for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error retrieving repositories: {str(e)}"
        )

# ===== USER SYNC STATUS =====

@user_router.get(
    "/{user_id}/sync-status", 
    response_model=UserSyncStatusResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user_sync_status(
    user_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Get sync status and statistics for a user
    """
    logger.info(f"Fetching sync status for user_id: {user_id}")
    try:
        # Check if user exists
        logger.debug(f"Verifying user existence - user_id: {user_id}")
        result = await db.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            logger.warning(f"User not found while fetching sync status - user_id: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )

        # Get GitHub accounts count
        accounts_result = await db.execute(
            select(func.count(UserGitHubAccount.id))
            .where(UserGitHubAccount.user_id == user_id)
        )
        total_accounts = accounts_result.scalar()
        logger.debug(f"Total GitHub accounts: {total_accounts}")

        # Get repositories count
        repos_result = await db.execute(
            select(func.count(UserRepository.id))
            .join(UserGitHubAccount)
            .where(UserGitHubAccount.user_id == user_id)
        )
        total_repositories = repos_result.scalar()
        logger.debug(f"Total repositories: {total_repositories}")

        # Get branches count
        branches_result = await db.execute(
            select(func.count(UserBranch.id))
            .join(UserRepository)
            .join(UserGitHubAccount)
            .where(UserGitHubAccount.user_id == user_id)
        )
        total_branches = branches_result.scalar()
        logger.debug(f"Total branches: {total_branches}")

        # Get sync jobs statistics
        sync_jobs_result = await db.execute(
            select(
                func.count(SyncJob.id),
                func.count(SyncJob.id).filter(SyncJob.status == "completed"),
                func.count(SyncJob.id).filter(SyncJob.status == "failed"),
                func.count(SyncJob.id).filter(SyncJob.status == "running")
            )
            .where(SyncJob.user_id == user_id)
        )
        sync_stats = sync_jobs_result.first()
        
        total_sync_jobs, completed_jobs, failed_jobs, running_jobs = sync_stats
        logger.debug(f"Sync jobs - total: {total_sync_jobs}, completed: {completed_jobs}, failed: {failed_jobs}, running: {running_jobs}")

        # Get recent sync jobs
        recent_jobs_result = await db.execute(
            select(SyncJob)
            .where(SyncJob.user_id == user_id)
            .order_by(SyncJob.created_at.desc())
            .limit(10)
        )
        recent_jobs = recent_jobs_result.scalars().all()

        recent_jobs_list = []
        for job in recent_jobs:
            recent_jobs_list.append(SyncJobSummary(
                id=job.id,
                job_type=job.job_type,
                status=job.status,
                trigger_event=job.trigger_event,
                started_at=job.started_at,
                completed_at=job.completed_at,
                created_at=job.created_at
            ))

        logger.info(f"Successfully compiled sync status for user {user_id}")
        return UserSyncStatusResponse(
            user_id=user_id,
            statistics=SyncStatistics(
                github_accounts=total_accounts,
                repositories=total_repositories,
                branches=total_branches,
                total_sync_jobs=total_sync_jobs,
                completed_jobs=completed_jobs,
                failed_jobs=failed_jobs,
                running_jobs=running_jobs
            ),
            recent_sync_jobs=recent_jobs_list
        )
        
    except HTTPException:
        logger.warning(f"HTTP exception while fetching sync status for user {user_id}")
        raise
    except Exception as e:
        logger.error(f"Error retrieving sync status for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error retrieving sync status: {str(e)}"
        )

# ===== USER ACTIVITY =====

@user_router.get(
    "/{user_id}/activity", 
    response_model=UserActivityResponse,
    responses={500: {"model": ErrorResponse}}
)
async def get_user_activity(
    user_id: int,
    limit: int = Query(50, ge=1, le=200),
    db: AsyncSession = Depends(get_async_db)
):
    """
    Get recent activity for a user (sync jobs, code changes, etc.)
    """
    logger.info(f"Fetching user activity for user_id: {user_id}, limit: {limit}")
    try:
        # Get recent code changes
        logger.debug(f"Querying code changes for user {user_id} with limit {limit}")
        code_changes = await database_service.get_code_changes_by_user(db, user_id, limit)
        
        changes_list = []
        for change in code_changes:
            changes_list.append(ActivitySummary(
                id=change.id,
                repo=change.repo,
                branch=change.branch,
                commit_hash=change.commit_hash,
                commit_author=change.commit_author,
                commit_timestamp=change.commit_timestamp,
                total_files_changed=change.total_files_changed,
                files_added=change.files_added,
                files_modified=change.files_modified,
                files_deleted=change.files_deleted,
                created_at=change.created_at
            ))

        logger.debug(f"Retrieved {len(changes_list)} activity items for user {user_id}")
        return UserActivityResponse(
            user_id=user_id,
            total_activities=len(changes_list),
            recent_activities=changes_list
        )
        
    except Exception as e:
        logger.error(f"Error retrieving user activity for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error retrieving user activity: {str(e)}"
        )

# ===== HEALTH CHECK =====

@user_router.get(
    "/{user_id}/health", 
    response_model=UserHealthResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user_health(
    user_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Get health status of user's GitHub installations and sync capabilities
    """
    logger.info(f"Checking health status for user_id: {user_id}")
    try:
        # Check if user exists
        logger.debug(f"Verifying user existence - user_id: {user_id}")
        result = await db.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            logger.warning(f"User not found while checking health - user_id: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )
        
        result = await db.execute(
            select(UserGitHubAccount)
            .where(UserGitHubAccount.user_id == user_id)
            .options(selectinload(UserGitHubAccount.repositories))
        )
        accounts = result.scalars().all()

        health_status = []
        for account in accounts:
            # Get repository count directly from the loaded relationship
            repositories_count = len(account.repositories) if account.repositories else 0
            
            account_health = AccountHealth(
                github_username=account.github_username,
                installation_id=account.installation_id,
                is_active=account.is_active,
                suspended=account.suspended_at is not None,
                repositories_count=repositories_count,
                last_updated=account.updated_at
            )
            health_status.append(account_health)

        # Determine overall health
        if all(acc.is_active and not acc.suspended for acc in health_status):
            overall_health = "healthy"
            logger.debug(f"User {user_id} health status: healthy")
        elif any(acc.is_active and not acc.suspended for acc in health_status):
            overall_health = "degraded"
            logger.warning(f"User {user_id} health status: degraded")
        else:
            overall_health = "unhealthy"
            logger.error(f"User {user_id} health status: unhealthy")

        logger.info(f"Health check completed for user {user_id} - overall: {overall_health}")
        return UserHealthResponse(
            user_id=user_id,
            health_status=health_status,
            overall_health=overall_health
        )
        
    except HTTPException:
        logger.warning(f"HTTP exception while checking health for user {user_id}")
        raise
    except Exception as e:
        logger.error(f"Error checking user health for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error checking user health: {str(e)}"
        )

@user_router.get(
    "/{user_id}/webhooks",
    response_model=WebhookListResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user_webhooks(
    user_id: int,
    limit: int = Query(20, ge=1, le=100, description="Number of webhooks to return"),
    db: AsyncSession = Depends(get_async_db)
):
    """Get webhook history for a user."""
    logger.info(f"Fetching webhooks for user_id: {user_id}, limit: {limit}")
    try:
        # Verify user exists
        logger.debug(f"Verifying user existence - user_id: {user_id}")
        user = await database_service.get_user_by_id(db, user_id)
        if not user:
            logger.warning(f"User not found while fetching webhooks - user_id: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )
        
        # Get webhooks
        logger.debug(f"Querying webhooks for user {user_id} with limit {limit}")
        webhooks = await database_service.get_webhooks_by_user_id(db, user_id, limit=limit)
        
        # Format response
        formatted_webhooks = []
        for webhook in webhooks:
            webhook: Webhook
            formatted_webhooks.append(WebhookResponse(
                id=webhook.id,
                repository_id=webhook.repository_id,
                webhook_id=webhook.id,
                event_type=webhook.event_type,
                status=webhook.status,
                delivery_id=webhook.delivery_id,
                signature=webhook.signature,
                created_at=webhook.created_at,
                processed_at=webhook.processed_at,
                payload=webhook.payload
            ))
        
        total_count = len(formatted_webhooks)
        logger.debug(f"Retrieved {total_count} webhooks for user {user_id}")

        return WebhookListResponse(
            user_id=user_id,
            total_count=total_count,
            webhooks=formatted_webhooks
        )
        
    except HTTPException:
        logger.warning(f"HTTP exception while fetching webhooks for user {user_id}")
        raise
    except Exception as e:
        logger.error(f"Error getting webhooks for user {user_id}: {str(e)}", exc_info=True)
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error fetching webhooks"
        )