from fastapi import FastAPI
from app.core.config import settings
from fastapi.middleware.cors import CORSMiddleware
from app.middleware.logger import LoggingMiddleware
from app.middleware.responses import json_response
import psutil
from app.services.kafka.kafka_client import on_app_startup, on_app_shutdown
import asyncio


app = FastAPI(title=settings.APP_NAME)

# CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=[settings.APP_CORS_ORIGINS],  # Adjust as needed for production
    allow_credentials=True,
    allow_methods=[settings.APP_CORS_METHODS],
    allow_headers=[settings.APP_CORS_HEADERS],
)
app.add_middleware(LoggingMiddleware)



@app.get("/")
async def root():
    return json_response(status_code=200, message=f"Welcome to the {settings.APP_NAME}", data={"app_name": settings.APP_NAME})

@app.get("/health")
async def health_check():
    # CPU overall usage
    cpu_percent = psutil.cpu_percent(interval=0.5)

    # Per-core CPU usage
    cpu_per_core = psutil.cpu_percent(interval=0.5, percpu=True)

    # CPU details
    cpu_count_logical = psutil.cpu_count(logical=True)
    cpu_count_physical = psutil.cpu_count(logical=False)
    cpu_freq = psutil.cpu_freq()

    # Memory stats
    memory = psutil.virtual_memory()
    ram_total = round(memory.total / (1024 ** 3), 2)  # GB
    ram_available = round(memory.available / (1024 ** 3), 2)  # GB
    ram_used = round(memory.used / (1024 ** 3), 2)  # GB
    ram_percent = memory.percent  # used %

    # Build response content
    content = {
        "cpu": {
            "overall_used_percent": cpu_percent,
            "per_core_percent": cpu_per_core,
            "physical_cores": cpu_count_physical,
            "logical_cores": cpu_count_logical,
            "frequency": {
                "current_mhz": cpu_freq.current if cpu_freq else None,
                "min_mhz": cpu_freq.min if cpu_freq else None,
                "max_mhz": cpu_freq.max if cpu_freq else None,
            }
        },
        "ram": {
            "total_gb": ram_total,
            "available_gb": ram_available,
            "used_gb": ram_used,
            "used_percent": ram_percent
        }
    }
    return json_response(status_code=200, message="API is healthy",data=content)

@app.get("/health/kafka")
async def health_check():
    from app.services.kafka.kafka_client import kafka_client
    kafka_healthy = await kafka_client.health_check()
    return {
        "status": "healthy",
        "kafka": "connected" if kafka_healthy else "disconnected"
    }


@app.on_event("startup")
async def startup_event():
    await on_app_startup()

@app.on_event("shutdown")
async def shutdown_event():
    await on_app_shutdown()
    # Give some time for graceful shutdown
    await asyncio.sleep(2)

from app.api.v1.routes import v1_router
app.include_router(v1_router, prefix="/api/v1")
