from pydantic_settings import BaseSettings
from pydantic import Field
from typing import Optional
from pathlib import Path


class Settings(BaseSettings):

    APP_DOMAIN: str = Field(..., env="APP_DOMAIN")
    # App
    APP_NAME: str = Field("apparchaea-repo-sync", env="APP_NAME")

    # Logging
    APP_LOG_DIR: str = Field("logs", env="APP_LOG_DIR")
    APP_LOG_FILE_NAME: str = Field("app.log", env="APP_LOG_FILE_NAME")
    APP_LOG_LEVEL: str = Field("info", env="APP_LOG_LEVEL")
    APP_LOG_BACKUP_DAYS: int = Field(360, env="APP_LOG_BACKUP_DAYS")

    # Server
    APP_HOST: str = Field("0.0.0.0", env="APP_HOST")
    APP_PORT: int = Field(9000, env="APP_PORT")
    APP_RELOAD: bool = Field(True, env="APP_RELOAD")
    APP_WORKERS: int = Field(1, env="APP_WORKERS")
    APP_DEBUG: bool = Field(True, env="APP_DEBUG")

    # CORS
    APP_CORS_ORIGINS: str = Field("*", env="APP_CORS_ORIGINS")
    APP_CORS_METHODS: str = Field("*", env="APP_CORS_METHODS")
    APP_CORS_HEADERS: str = Field("*", env="APP_CORS_HEADERS")

    # GitHub OAuth / Webhooks
    GITHUB_WEBHOOK_SECRET: str = Field("", env="GITHUB_WEBHOOK_SECRET")
    GITHUB_CLIENT_ID: str = Field("", env="GITHUB_CLIENT_ID")
    GITHUB_CLIENT_SECRET: str = Field("", env="GITHUB_CLIENT_SECRET")
    GITHUB_CALLBACK_URL: str = Field("", env="GITHUB_CALLBACK_URL")
    GITHUB_APP_ID: str = Field("", env="GITHUB_APP_ID")
    GITHUB_PRIVATE_KEY_NAME: str = Field("", env="GITHUB_PRIVATE_KEY_NAME")  # just the filename

    @property
    def GITHUB_PRIVATE_KEY(self) -> str:
        """Load GitHub private key from project root using only filename."""
        key_path = Path(__file__).parent.parent.parent / self.GITHUB_PRIVATE_KEY_NAME
        if not key_path.exists():
            raise FileNotFoundError(f"GitHub private key file not found: {key_path}")
        return key_path.read_text()

    # Redis settings
    REDIS_HOST: str = Field(..., env="REDIS_HOST")
    REDIS_PORT: int = Field(..., env="REDIS_PORT")
    REDIS_DB: int = Field(..., env="REDIS_DB")
    REDIS_PASSWORD: Optional[str] = Field(None, env="REDIS_PASSWORD")

    # Storage Configuration
    STORAGE_TYPE: str = Field(..., env="STORAGE_TYPE")
    LOCAL_STORAGE_PATH: str = Field(..., env="LOCAL_STORAGE_PATH")

    # AWS S3 Configuration
    S3_BUCKET_NAME: str = Field(..., env="S3_BUCKET_NAME")
    S3_REGION: str = Field(..., env="S3_REGION")
    AWS_ACCESS_KEY_ID: str = Field(..., env="AWS_ACCESS_KEY_ID")
    AWS_SECRET_ACCESS_KEY: str = Field(..., env="AWS_SECRET_ACCESS_KEY")
    S3_PRESIGNED_URL_EXPIRY: int = Field(..., env="S3_PRESIGNED_URL_EXPIRY")
    S3_ENDPOINT_URL: str = Field("fsn1.your-objectstorage.com", env="S3_ENDPOINT_URL")


    # PostgreSQL Database Configuration
    POSTGRES_HOST: str = Field("localhost", env="POSTGRES_HOST")
    POSTGRES_PORT: int = Field(5432, env="POSTGRES_PORT")
    POSTGRES_USER: str = Field("postgres", env="POSTGRES_USER")
    POSTGRES_PASSWORD: str = Field("password", env="POSTGRES_PASSWORD")
    POSTGRES_DB: str = Field("archaea-repo-sync", env="POSTGRES_DB")
    DB_ECHO: bool = Field(False, env="DB_ECHO")

    # --- Constants (now configurable via .env) ---
    INSTALLATION_WAIT_TIMEOUT: int = Field(60, env="INSTALLATION_WAIT_TIMEOUT")  # seconds
    WEBHOOK_PROCESSING_DELAY: int = Field(2, env="WEBHOOK_PROCESSING_DELAY")      # seconds
    MAX_WEBHOOK_PAYLOAD_SIZE: int = Field(10 * 1024 * 1024, env="MAX_WEBHOOK_PAYLOAD_SIZE")  # 10MB

    def get_postgres_dsn(self) -> str:
        """Async PostgreSQL DSN for SQLAlchemy"""
        return f"postgresql+asyncpg://{self.POSTGRES_USER}:{self.POSTGRES_PASSWORD}@{self.POSTGRES_HOST}:{self.POSTGRES_PORT}/{self.POSTGRES_DB}"

    def get_sync_postgres_dsn(self) -> str:
        """Synchronous PostgreSQL DSN for SQLAlchemy"""
        return f"postgresql://{self.POSTGRES_USER}:{self.POSTGRES_PASSWORD}@{self.POSTGRES_HOST}:{self.POSTGRES_PORT}/{self.POSTGRES_DB}"

    def get_syncpsycopg2_postgres_dsn(self) -> str:
        """Synchronous PostgreSQL DSN using psycopg2"""
        return f"postgresql+psycopg2://{self.POSTGRES_USER}:{self.POSTGRES_PASSWORD}@{self.POSTGRES_HOST}:{self.POSTGRES_PORT}/{self.POSTGRES_DB}"
    # Kafka Configuration
    KAFKA_BROKER: str = Field("localhost:9092", env="KAFKA_BROKER")
    KAFKA_USER: Optional[str] = Field(None, env="KAFKA_USER")
    KAFKA_PASSWORD: Optional[str] = Field(None, env="KAFKA_PASSWORD")
    KAFKA_TOPIC: str = Field("events", env="KAFKA_TOPIC")
    KAFKA_GROUP_ID: str = Field("my-group", env="KAFKA_GROUP_ID")
    KAFKA_SECURITY_PROTOCOL: str = Field("PLAINTEXT", env="KAFKA_SECURITY_PROTOCOL")
    KAFKA_SASL_MECHANISM: Optional[str] = Field(None, env="KAFKA_SASL_MECHANISM")

    # Multiple topics for different message types
    KAFKA_WEBHOOK_TOPIC: str = "webhooks"
    KAFKA_SYNC_TOPIC: str = "sync"
    KAFKA_PUSH_TOPIC: str = "push"
    KAFKA_AST_TOPIC: str = "ast_processing"
    KAFKA_CHUNKING_TOPIC: str = "chunking"

    def get_kafka_config(self) -> dict:
        """Return Kafka client configuration for confluent_kafka or aiokafka"""
        config = {
            "bootstrap_servers": self.KAFKA_BROKER,
            "security_protocol": self.KAFKA_SECURITY_PROTOCOL,
        }
        if self.KAFKA_USER and self.KAFKA_PASSWORD:
            config.update({
                "sasl_mechanism": self.KAFKA_SASL_MECHANISM or "PLAIN",
                "sasl_plain_username": self.KAFKA_USER,
                "sasl_plain_password": self.KAFKA_PASSWORD,
            })
        return config

    class Config:
        env_file = ".env"
        case_sensitive = True


settings = Settings()
