import logging
import os
import uuid
from logging.handlers import TimedRotatingFileHandler
from datetime import datetime, timezone
from contextvars import ContextVar
from app.core.config import settings

# Context variable to store request ID for the current request
request_id_ctx: ContextVar[str] = ContextVar('request_id', default='no-request-id')


class RequestIDFilter(logging.Filter):
    """Log filter to inject request ID into log records."""
    
    def filter(self, record):
        record.request_id = request_id_ctx.get()
        return True


class UTCFormatter(logging.Formatter):
    """Formatter that ensures timestamps are in UTC and includes request ID."""
    converter = datetime.fromtimestamp

    def formatTime(self, record, datefmt=None):
        dt = datetime.fromtimestamp(record.created, tz=timezone.utc)
        if datefmt:
            return dt.strftime(datefmt)
        return dt.isoformat()

    def format(self, record):
        # Ensure request_id attribute exists
        if not hasattr(record, 'request_id'):
            record.request_id = 'no-request-id'
        return super().format(record)


def setup_logger(name: str = "app") -> logging.Logger:
    """
    Setup and return a logger that:
      - Logs in UTC
      - Rotates daily
      - Keeps up to settings.LOG_BACKUP_DAYS
      - Includes request ID in every log
    """

    logger = logging.getLogger(name)
    if logger.handlers:
        return logger  # already configured

    # Create log directory if not exists
    os.makedirs(settings.APP_LOG_DIR, exist_ok=True)

    # Log file path
    log_file = os.path.join(settings.APP_LOG_DIR, settings.APP_LOG_FILE_NAME)

    # File handler (daily rotation, UTC, keep N days)
    file_handler = TimedRotatingFileHandler(
        log_file,
        when="midnight",
        interval=1,
        backupCount=settings.APP_LOG_BACKUP_DAYS,
        utc=True,
    )

    # Updated formatter to include request ID
    formatter = UTCFormatter(
        fmt="%(asctime)s | %(levelname)s | %(name)s | Request-ID: %(request_id)s | %(message)s",
        datefmt="%Y-%m-%dT%H:%M:%SZ",
    )

    file_handler.setFormatter(formatter)

    # Console handler
    console_handler = logging.StreamHandler()
    console_handler.setFormatter(formatter)

    # Add request ID filter to both handlers
    request_id_filter = RequestIDFilter()
    file_handler.addFilter(request_id_filter)
    console_handler.addFilter(request_id_filter)

    # Attach handlers
    logger.setLevel(settings.APP_LOG_LEVEL.upper())
    logger.addHandler(file_handler)
    logger.addHandler(console_handler)
    logger.propagate = False

    return logger


# Global logger accessible anywhere
logger = setup_logger()


# Utility function to get current request ID
def get_request_id() -> str:
    """Get the current request ID from context."""
    return request_id_ctx.get()


# Function to set request ID (used by middleware)
def set_request_id(request_id: str):
    """Set the current request ID in context."""
    request_id_ctx.set(request_id)