# app/models/github_models.py
from sqlalchemy import Column, Integer, String, BigInteger, Boolean, DateTime, Text, JSON, ForeignKey, UniqueConstraint
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func
from app.db.postgress_db import Base

class UserRepository(Base):
    __tablename__ = "user_repositories"
    
    id = Column(Integer, primary_key=True, index=True)
    github_account_id = Column(Integer, ForeignKey("user_github_accounts.id", ondelete="CASCADE"), nullable=False)
    repo_id = Column(BigInteger, nullable=False)  # GitHub repository ID
    name = Column(String(255), nullable=False)
    full_name = Column(String(255), nullable=False)
    private = Column(Boolean, default=False)
    html_url = Column(Text)
    description = Column(Text)
    language = Column(String(100))
    default_branch = Column(String(255), default="main")
    is_active = Column(Boolean, default=True)
    last_synced_at = Column(DateTime(timezone=True), nullable=True)
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    updated_at = Column(DateTime(timezone=True), onupdate=func.now())
    
    # Relationships
    github_account = relationship("UserGitHubAccount", back_populates="repositories")
    branches = relationship("UserBranch", back_populates="repository")
    webhooks = relationship("Webhook", back_populates="repository")
    branch_files = relationship("UserBranchFile", back_populates="repository")  

    
    # Unique constraint
    __table_args__ = (
        UniqueConstraint('github_account_id', 'repo_id', name='uq_account_repo'),
    )

class UserBranch(Base):
    __tablename__ = "user_branches"
    
    id = Column(Integer, primary_key=True, index=True)
    repository_id = Column(Integer, ForeignKey("user_repositories.id", ondelete="CASCADE"), nullable=False)
    name = Column(String(255), nullable=False)
    commit_sha = Column(String(255), nullable=False)
    commit_message = Column(Text)
    commit_author = Column(String(255))
    commit_timestamp = Column(DateTime(timezone=True))
    is_protected = Column(Boolean, default=False)
    last_synced_at = Column(DateTime(timezone=True), nullable=True)
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    updated_at = Column(DateTime(timezone=True), onupdate=func.now())
    
    # Relationships
    repository = relationship("UserRepository", back_populates="branches")
    sync_jobs = relationship("SyncJob", back_populates="branch")
    branch_files = relationship("UserBranchFile", back_populates="branch") 

    
    # Unique constraint
    __table_args__ = (
        UniqueConstraint('repository_id', 'name', name='uq_repo_branch'),
    )

class UserBranchFile(Base):
    __tablename__ = "user_branch_files"
    
    id = Column(Integer, primary_key=True, index=True)
    repository_id = Column(Integer, ForeignKey("user_repositories.id", ondelete="CASCADE"), nullable=False)
    branch_id = Column(Integer, ForeignKey("user_branches.id", ondelete="CASCADE"), nullable=False)
    file_id = Column(String(255), nullable=False)
    file_path = Column(Text, nullable=False)
    file_name = Column(String(255), nullable=False)
    file_extension = Column(String(50))
    file_size = Column(BigInteger)
    sha = Column(String(255), nullable=False)  # Git SHA - use this to fetch content from GitHub API when needed
    file_status = Column(String(50), default="active")  # active, modified, deleted, moved, renamed
    user_id = Column(Integer, ForeignKey("users.id", ondelete="CASCADE"), nullable=False)

    last_modified = Column(DateTime(timezone=True))
    is_active = Column(Boolean, default=True)
    updated_at = Column(DateTime(timezone=True), server_default=func.now())

    ast_storage_path = Column(Text, nullable=True)  # Local or remote path (S3, GCS, etc.)
    ast_processed_at = Column(DateTime(timezone=True), default=None)  # When AST was generated

    
    # Relationships
    repository = relationship("UserRepository", back_populates="branch_files")
    branch = relationship("UserBranch", back_populates="branch_files")

class Webhook(Base):
    __tablename__ = "webhooks"
    
    id = Column(Integer, primary_key=True, index=True)
    repository_id = Column(Integer, ForeignKey("user_repositories.id", ondelete="CASCADE"), nullable=True)
    branch_id = Column(Integer, ForeignKey("user_branches.id", ondelete="SET NULL"), nullable=True)
    github_webhook_id = Column(BigInteger, nullable=False)  # GitHub webhook ID
    event_type = Column(String(100), nullable=False)  # push, installation, etc.
    payload = Column(JSON, nullable=False)
    headers = Column(JSON)
    delivery_id = Column(String(255), unique=True)  # GitHub delivery ID
    signature = Column(String(255))
    status = Column(String(50), default="received")  # received, processed, failed
    processed_at = Column(DateTime(timezone=True), nullable=True)
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    user_id = Column(Integer, ForeignKey("users.id", ondelete="CASCADE"), nullable=True)
    installation_id = Column(BigInteger, nullable=True)  # GitHub App installation ID
    error_log = Column(Text, nullable=True)  # store error messages or processing logs



    # Relationships
    repository = relationship("UserRepository", back_populates="webhooks")
    branch = relationship("UserBranch")  


     # Unique constraints with installation_id
    __table_args__ = (
        # Each webhook delivery should be unique per installation
        UniqueConstraint('installation_id', 'delivery_id', name='uq_installation_delivery'),
    )