from pydantic import BaseModel, Field, validator
from typing import List, Optional, Dict, Any
from datetime import datetime
from enum import Enum

# ===== ENUMS =====
class EventType(str, Enum):
    INSTALLATION = "installation"
    PUSH = "push"
    PULL_REQUEST = "pull_request"
    ISSUES = "issues"
    UNKNOWN = "unknown"

class AccountType(str, Enum):
    USER = "User"
    ORGANIZATION = "Organization"
    BOT = "Bot"

class RepositoryVisibility(str, Enum):
    PUBLIC = "public"
    PRIVATE = "private"
    INTERNAL = "internal"

class SyncType(str, Enum):
    AUTO_INSTALL = "auto_install"
    PUSH_FULL_BRANCH = "push_full_branch"
    PUSH_CHANGED_FILES = "push_changed_files"
    PUSH_CHANGED_FILES_DIFF = "push_changed_files_diff"
    MANUAL_SYNC = "manual_sync"

class WebhookStatus(str, Enum):
    RECEIVED = "received"
    PROCESSING = "processing"
    PROCESSED = "processed"
    FAILED = "failed"
    UNKNOWN_EVENT = "unknown_event"

class SyncJobStatus(str, Enum):
    RUNNING = "running"
    COMPLETED = "completed"
    FAILED = "failed"
    PENDING = "pending"

# ===== BASE SCHEMAS =====
class GitHubAccountBase(BaseModel):
    github_username: str = Field(..., min_length=1, max_length=255)
    github_id: int = Field(..., gt=0)
    installation_id: int = Field(..., gt=0)
    account_type: AccountType = AccountType.USER
    avatar_url: Optional[str] = None
    html_url: Optional[str] = None

# ===== CREATE/UPDATE SCHEMAS =====
class GitHubAccountCreate(GitHubAccountBase):
    user_id: int = Field(..., gt=0)
    permissions: Optional[Dict[str, Any]] = None
    is_active: bool = True

# ===== REQUEST SCHEMAS =====
class GitHubInstallRequest(BaseModel):
    user_id: int = Field(..., gt=0)
    

class TestWebhookRequest(BaseModel):
    event: str = Field(..., description="Webhook event type")
    payload: Dict[str, Any] = Field(..., description="Webhook payload")


class ManualSyncRequest(BaseModel):
    user_id: int = Field(..., description="Internal user ID")

# ===== RESPONSE SCHEMAS =====
class GitHubAccountResponse(BaseModel):
    id: int
    user_id: int
    github_username: str
    github_id: int
    installation_id: int
    account_type: AccountType
    avatar_url: Optional[str]
    html_url: Optional[str]
    is_active: bool
    permissions: Optional[Dict[str, Any]]
    suspended_at: Optional[datetime]
    created_at: datetime
    updated_at: Optional[datetime]

    class Config:
        from_attributes = True

class GitHubAccountCreateResponse(BaseModel):
    message: str
    account_id: int
    github_username: str
    installation_id: int

class GitHubAccountListResponse(BaseModel):
    user_id: int
    total_accounts: int
    github_accounts: List[GitHubAccountResponse]

class RepositoryResponse(BaseModel):
    id: int
    github_account_id: int
    repo_id: int
    name: str
    full_name: str
    private: bool
    html_url: Optional[str]
    description: Optional[str]
    language: Optional[str]
    default_branch: str
    is_active: bool
    last_synced_at: Optional[datetime]
    created_at: datetime
    updated_at: Optional[datetime]

    class Config:
        from_attributes = True

class RepositoryListResponse(BaseModel):
    user_id: int
    github_username: Optional[str]
    total_repositories: int
    repositories: List[RepositoryResponse]


# ===== API RESPONSE SCHEMAS =====
class GitHubCallbackResponse(BaseModel):
    message: str = Field(..., description="Response message")
    setup_action: Optional[str] = Field(None, description="Setup action from GitHub")
    install_url: Optional[str] = Field(None, description="GitHub App installation URL")

    class Config:
        from_attributes = True

class ManualSyncResponse(BaseModel):
    message: str = Field(..., description="Sync status message")
    user_id: int = Field(..., description="Internal user ID")
    installation_id: int = Field(..., description="GitHub App installation ID")
    webhook_id: int = Field(..., description="Created sync job ID")
    timestamp: datetime = Field(default_factory=datetime.now, description="Sync timestamp")

class WebhookTestResponse(BaseModel):
    status: str = Field(..., description="Test status")
    event: str = Field(..., description="Webhook event type")
    user_id: int = Field(..., description="Internal user ID")
    message: str = Field(..., description="Response message")
    actions_performed: List[str] = Field(..., description="List of actions performed")
    webhook_id: Optional[int] = Field(None, description="Created webhook ID")
    notes: List[str] = Field(..., description="Additional notes")

class DownloadResponse(BaseModel):
    repo_id: int = Field(..., description="Repository ID")
    repo_full_name: str = Field(..., description="Repository full name")
    repo_name: str = Field(..., description="Repository name")
    user_id: str = Field(..., description="Internal user ID")
    ref: str = Field(..., description="Branch/tag/commit reference")
    format: str = Field(..., description="Archive format")
    download_url: str = Field(..., description="Download URL")
    storage_info: 'StorageInfo' = Field(..., description="Storage information")
    file_size: int = Field(..., description="File size in bytes")
    message: str = Field(..., description="Response message")

class TestInstallationResponse(BaseModel):
    accessible: bool = Field(..., description="Whether installation is accessible")
    installation_id: Optional[int] = Field(None, description="Installation ID if found")
    username: Optional[str] = Field(None, description="GitHub username")
    token_generated: bool = Field(..., description="Whether token was generated")
    error: Optional[str] = Field(None, description="Error message if any")

# ===== DATA TRANSFER SCHEMAS =====
class RepositoryInfo(BaseModel):
    id: int = Field(..., description="GitHub repository ID")
    name: str = Field(..., description="Repository name")
    full_name: str = Field(..., description="Repository full name (owner/repo)")
    private: bool = Field(..., description="Whether repository is private")
    html_url: str = Field(..., description="GitHub URL for the repository")
    description: Optional[str] = Field(None, description="Repository description")
    language: Optional[str] = Field(None, description="Primary programming language")
    updated_at: datetime = Field(..., description="Last update timestamp")
    visibility: str = Field(..., description="Repository visibility (public/private)")
    default_branch: str = Field(..., description="Default branch name")
    archived: bool = Field(False, description="Whether repository is archived")

    class Config:
        from_attributes = True

class RepositoryListConfigResponse(BaseModel):
    user_id: int = Field(..., description="Internal user ID")
    total_repositories: int = Field(..., ge=0, description="Total number of repositories")
    private_repositories: int = Field(..., ge=0, description="Number of private repositories")
    public_repositories: int = Field(..., ge=0, description="Number of public repositories")
    repositories: List[RepositoryInfo] = Field(..., description="List of repositories")

    class Config:
        from_attributes = True

class BranchInfo(BaseModel):
    id: int = Field(..., description="Branch ID in database")
    name: str = Field(..., description="Branch name")
    commit_sha: str = Field(..., min_length=7, max_length=40, description="Latest commit SHA")
    commit_message: Optional[str] = Field(None, description="Commit message")
    commit_author: Optional[str] = Field(None, description="Commit author")
    is_protected: bool = Field(False, description="Whether branch is protected")
    last_synced_at: Optional[datetime] = Field(None, description="Last sync timestamp")

    class Config:
        from_attributes = True

class BranchListConfResponse(BaseModel):
    repository_id: int = Field(..., description="Repository ID")
    repository_name: str = Field(..., description="Repository name")
    total_branches: int = Field(..., ge=0, description="Total number of branches")
    branches: List[BranchInfo] = Field(..., description="List of branches")

    class Config:
        from_attributes = True


class StorageInfo(BaseModel):
    storage_type: str = Field(..., description="Storage type (local/s3)")
    filename: str = Field(..., description="Stored filename")
    download_url: str = Field(..., description="Download URL")
    uploaded_at: datetime = Field(..., description="Upload timestamp")
    file_size: Optional[int] = Field(None, description="File size in bytes")