from sqlalchemy.ext.asyncio import AsyncSession
from app.core.logger import logger
from app.services.postgress_db_service import pg_db_service as database_service
from datetime import datetime
from typing import Dict, List, Any, Optional, Set, Tuple
from app.services.kafka.kafka_client import send_ast_processing_message, send_chunking_message, send_push_message, send_webhook_message, send_sync_message
import asyncio

class ASTFileSync:
    """AST Processor for handling file analysis and Kafka messaging."""
    

    async def _process_all_files_ast(
        self,
        file_ids: List[int],
        webhook_id: int,
        db: AsyncSession
    ) -> List[int]:
        """Process all files through AST - returns list of processed file_ids."""
        from app.services.webhook.webhook_manager import WebhookManager
        webhook_manger=WebhookManager()
        webhook_data = await webhook_manger._get_webhook_data(webhook_id, db)
        if not webhook_data:
            return []
        
        # Send all messages concurrently
        tasks = []
        for file_id in file_ids:
            kafka_message = {
                "file_id": file_id,
                "webhook_id": webhook_id,
                "event_type": "process_ast",
                "user_id": webhook_data["user_id"],
                "installation_id": webhook_data["installation_id"],
                "timestamp": datetime.utcnow().isoformat(),
                "source": "installation_pipeline"
            }
            
            task = send_ast_processing_message(
                file_id=file_id,
                data=kafka_message
            )
            tasks.append(task)

        # Wait for all messages to be sent and collect results
        results = await asyncio.gather(*tasks, return_exceptions=True)
        
        # Process results to get successful file_ids
        processed_file_ids = []
        for file_id, result in zip(file_ids, results):
            if isinstance(result, Exception):
                logger.error(f"Failed to send AST message for file {file_id}: {result}")
            elif result:  # If send_message returns True on success
                processed_file_ids.append(file_id)
            else:
                logger.warning(f"AST message sending failed for file {file_id}")
        
        logger.info(f"AST processing messages sent for {len(processed_file_ids)} files")
        return processed_file_ids

    async def process_files(
        self,
        file_ids: List[int],
        webhook_id: int,
        db: AsyncSession
    ) -> List[int]:
        """Public method to process files through AST."""
        return await self._process_all_files_ast(file_ids, webhook_id, db)

