# app/services/github/github_client.py
import httpx
import jwt
import time
from typing import Dict, List, Optional, Any
from app.core.logger import logger
from app.core.config import settings

class GitHubClient:
    """GitHub API client using simple httpx requests."""
    
    def __init__(self, installation_id: int = None):
        self.installation_id = installation_id
        self.base_url = "https://api.github.com"
        
        # Direct access to settings
        self.app_id = settings.GITHUB_APP_ID
        self.private_key = settings.GITHUB_PRIVATE_KEY
        self.webhook_secret = settings.GITHUB_WEBHOOK_SECRET
        self.app_name = settings.APP_NAME
        self.client_id = settings.GITHUB_CLIENT_ID
        self.client_secret = settings.GITHUB_CLIENT_SECRET
        self.oauth_base_url = "https://github.com"
    async def exchange_code_for_token(self, code: str) -> str:
        """Exchange OAuth code for access token."""
        try:
            async with httpx.AsyncClient() as client:
                data = {
                    "client_id": self.client_id,
                    "client_secret": self.client_secret,
                    "code": code
                }
                
                headers = {
                    "Accept": "application/json",
                    "User-Agent": self.app_name
                }
                
                response = await client.post(
                    f"{self.oauth_base_url}/login/oauth/access_token",
                    data=data,
                    headers=headers
                )
                
                if response.status_code == 200:
                    token_data = response.json()
                    access_token = token_data.get("access_token")
                    
                    if not access_token:
                        error_msg = token_data.get("error_description", "No access token in response")
                        logger.error(f"OAuth token exchange failed: {error_msg}")
                        return None
                    
                    logger.info("Successfully exchanged code for access token")
                    return access_token
                else:
                    logger.error(f"OAuth token exchange failed with status {response.status_code}: {response.text}")
                    return None
                    
        except Exception as e:
            logger.error(f"Failed to exchange code for token: {str(e)}")
            return None

    async def get_user_info(self, access_token: str) -> Dict:
        """Get user info using access token."""
        try:
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {access_token}",
                    "Accept": "application/vnd.github.v3+json",
                    "User-Agent": self.app_name
                }
                
                response = await client.get(
                    f"{self.base_url}/user",
                    headers=headers
                )
                
                if response.status_code == 200:
                    user_data = response.json()
                    logger.info(f"Retrieved user info for: {user_data.get('login')}")
                    return user_data
                else:
                    logger.error(f"Failed to get user info: {response.status_code} - {response.text}")
                    return None
                    
        except Exception as e:
            logger.error(f"Failed to get user info: {str(e)}")
            return None

    
    def _generate_jwt(self) -> str:
        """Generate JWT for GitHub App authentication."""
        try:
            payload = {
                "iat": int(time.time()),
                "exp": int(time.time()) + 600,
                "iss": self.app_id
            }
            return jwt.encode(payload, self.private_key, algorithm="RS256")
        except Exception as e:
            logger.error(f"Failed to generate JWT: {str(e)}")
            raise
    
    async def _get_installation_token(self) -> str:
        """Get installation access token."""
        try:
            jwt_token = self._generate_jwt()
            
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {jwt_token}",
                    "Accept": "application/vnd.github.v3+json",
                    "User-Agent": self.app_name
                }
                url = f"{self.base_url}/app/installations/{self.installation_id}/access_tokens"
                response = await client.post(url, headers=headers)
                
                if response.status_code == 201:
                    data = response.json()
                    return data["token"]
                else:
                    error_text = response.text
                    raise Exception(f"Failed to get installation token: {error_text}")
        except Exception as e:
            logger.error(f"Failed to get installation token: {str(e)}")
            raise
    
    async def _make_request(self, method: str, endpoint: str, **kwargs) -> Any:
        """Make authenticated request to GitHub API."""
        try:
            async with httpx.AsyncClient() as client:
                headers = kwargs.pop("headers", {})
                headers.update({
                    "Accept": "application/vnd.github.v3+json",
                    "User-Agent": self.app_name
                })
                
                if self.installation_id:
                    token = await self._get_installation_token()
                    headers["Authorization"] = f"token {token}"
                else:
                    jwt_token = self._generate_jwt()
                    headers["Authorization"] = f"Bearer {jwt_token}"
                
                url = f"{self.base_url}{endpoint}"
                response = await client.request(method, url, headers=headers, **kwargs)
                
                if response.status_code == 200:
                    return response.json()
                elif response.status_code == 204:
                    return None
                else:
                    error_text = response.text
                    logger.error(f"GitHub API error {response.status_code}: {error_text}")
                    raise Exception(f"GitHub API error {response.status_code}: {error_text}")
        except Exception as e:
            logger.error(f"GitHub API request failed: {str(e)}")
            raise
    async def _make_request_get_text(self, method: str, endpoint: str, **kwargs) -> str:
        """Make authenticated request to GitHub API - returns raw text."""
        try:
            async with httpx.AsyncClient() as client:
                headers = kwargs.pop("headers", {})
                headers.update({
                    "User-Agent": self.app_name
                })
                
                if self.installation_id:
                    token = await self._get_installation_token()
                    headers["Authorization"] = f"token {token}"
                else:
                    jwt_token = self._generate_jwt()
                    headers["Authorization"] = f"Bearer {jwt_token}"
                
                url = f"{self.base_url}{endpoint}"
                response = await client.request(method, url, headers=headers, **kwargs)
                
                if response.status_code == 200:
                    return response.text
                elif response.status_code == 204:
                    return ""
                else:
                    error_text = response.text
                    logger.error(f"GitHub API error {response.status_code}: {error_text}")
                    raise Exception(f"GitHub API error {response.status_code}: {error_text}")
        except Exception as e:
            logger.error(f"GitHub API request failed: {str(e)}")
            raise
    
    # Repository methods
    async def get_installation_repositories(self) -> List[Dict]:
        return await self._make_request("GET", "/installation/repositories")
    
    async def get_repository_branches(self, full_name: str) -> List[Dict]:
        return await self._make_request("GET", f"/repos/{full_name}/branches")
    
    async def get_branch_files(self, full_name: str, branch: str) -> List[Dict]:
        try:
            branch_data = await self.get_branch(full_name, branch)
            commit_sha = branch_data["commit"]["sha"]
            tree_data = await self._make_request(
                "GET", 
                f"/repos/{full_name}/git/trees/{commit_sha}",
                params={"recursive": "1"}
            )
            return tree_data.get("tree", [])
        except Exception as e:
            logger.error(f"Failed to get branch files: {str(e)}")
            return []
    
    async def get_branch(self, full_name: str, branch: str) -> Dict:
        return await self._make_request("GET", f"/repos/{full_name}/branches/{branch}")
    
    async def get_file_content(self, full_name: str, path: str, ref: str = "main") -> str:
        try:
            file_data = await self._make_request("GET", f"/repos/{full_name}/contents/{path}", params={"ref": ref})
            if file_data.get("encoding") == "base64":
                import base64
                return base64.b64decode(file_data["content"]).decode("utf-8")
            return file_data.get("content", "")
        except Exception as e:
            logger.error(f"Failed to get file content: {str(e)}")
            return ""
    
    def verify_webhook_signature(self, payload_body: bytes, signature_header: str) -> bool:
        """Verify GitHub webhook signature."""
        try:
            import hmac
            import hashlib
            if not signature_header.startswith('sha256='):
                return False
            signature = signature_header[7:]
            expected_signature = hmac.new(
                self.webhook_secret.encode(),
                payload_body,
                hashlib.sha256
            ).hexdigest()
            return hmac.compare_digest(signature, expected_signature)
        except Exception as e:
            logger.error(f"Webhook signature verification failed: {str(e)}")
            return False
        
    async def get_new_installation_user(self, user_id: str) -> str:
        """Generate installation URL for user."""
        install_url = (
            "https://github.com/apps/archaea-repo-sync/installations/new"
            f"?state={user_id}"
        )
        return install_url
    
    async def get_installation_info(self, installation_id: int) -> Dict[str, Any]:
        """Fetch installation information from GitHub API."""
        try:
            installation_info = await self._make_request(
                "GET", 
                f"/app/installations/{installation_id}"
            )
            return installation_info
        except Exception as e:
            logger.error(f"Error fetching installation info for {installation_id}: {e}")
            raise

    
    

github_clinet=GitHubClient()