# app/services/ast/kafka_ast_consumer.py
import asyncio
import json
from sqlalchemy.ext.asyncio import AsyncSession
from app.core.logger import logger
from app.db.postgress_db import get_async_db
from app.services.postgress_db_service import pg_db_service as database_service
from app.services.storage_service import storage_service
from app.services.ast.ast_json_generator import ASTProcessor
from datetime import datetime
from typing import Optional
from io import BytesIO


async def process_ast_message(message_data: dict):
    """Process AST messages from Kafka - generate and store AST JSON"""
    try:
        logger.info(f"🔮 Processing AST file_id: {message_data.get('file_id')}")
        
        file_id = message_data.get('file_id')
        if not file_id:
            logger.error("❌ No file_id in AST message")
            return
        webhook_id=message_data.get('webhook_id')
            
        async for db in get_async_db():
            try:
                # Get file details from database
                file_data = await database_service.get_file_by__id(db, file_id)
                if not file_data:
                    logger.error(f"❌ File not found: {file_id}")
                    return
                
                # Get file content
                file_content = await _get_file_content(file_data, db)
                if not file_content:
                    logger.error(f"❌ Could not get content for file: {file_data.file_path}")
                    return
                
                # Process AST using the AST processor
                ast_json = await _generate_ast_json(file_data, file_content, db)
                # Store AST JSON
                upload_result = await _store_ast_json(ast_json, file_data)
                
                # Update file record with AST info
                await database_service.update_file(db, file_data.id, {
                    "ast_storage_path": upload_result['normal_url'],
                    "ast_processed_at": datetime.utcnow(),
                })
                
                logger.info(f"✅ AST processed and stored for file-id: {file_id} webhook-id : {webhook_id}")
                
            except Exception as e:
                logger.error(f"❌ Error processing AST for file {file_id}: {str(e)}")
        
    except Exception as e:
        logger.error(f"❌ Fatal error processing AST message: {str(e)}")

async def _get_file_content(file_record, db: AsyncSession) -> Optional[str]:
    """Get file content from storage or GitHub."""
    try:
        from app.services.github.github_client import GitHubClient
        
        branch = await database_service.get_branch_by_id(db, file_record.branch_id)
        if not branch:
            return None
        
        repository = await database_service.get_repository_by_id(db, branch.repository_id)
        if not repository:
            return None
        
        github_account = await database_service.get_github_account_by_installation_id(
            db, repository.github_account.installation_id
        )
        
        github_client = GitHubClient(github_account.installation_id)
        content = await github_client.get_file_content(
            repository.full_name,
            file_record.file_path,
            branch.name
        )
        
        return content
        
    except Exception as e:
        logger.error(f"Failed to get file content for {file_record.file_path}: {str(e)}")
        return None

async def _generate_ast_json(file_data, file_content: str, db: AsyncSession) -> dict:
    """Generate AST JSON structure for a file using ASTProcessor."""
    try:
        # Initialize AST processor
        ast_processor = ASTProcessor()
        
        # Get user_id, repo_id, branch_id from the file data
        file_data = await database_service.get_file_by_file_id(db, file_data.file_id)

        # Process file with AST processor
        ast_result = await ast_processor.process_file_to_unified_json(
            file_content=file_content,
            file_path=file_data.file_path,
            file_id=file_data.file_id,
            user_id=file_data.user_id,
            repo_id=file_data.repository_id,
            branch_id=file_data.branch_id
        )
        
        # The AST processor already returns the complete structure
        # Just ensure cyclomatic_complexity is removed from metrics
        if "analysis" in ast_result and "metrics" in ast_result["analysis"]:
            # Remove cyclomatic_complexity if it exists
            ast_result["analysis"]["metrics"].pop("cyclomatic_complexity", None)
        
        return ast_result
        
    except Exception as e:
        logger.error(f"Error generating AST JSON: {str(e)}")
        # Return empty structure on error
        return _create_empty_ast_json(file_data, db)

async def _create_empty_ast_json(file_data, db: AsyncSession) -> dict:
    """Create empty AST JSON structure when processing fails."""
    try:
        branch = await database_service.get_branch_by_id(db, file_data.branch_id)
        repository = await database_service.get_repository_by_id(db, branch.repository_id)
        
        return {
            "hierarchy": {
                "user_id": repository.user_id,
                "repo_id": repository.id,
                "branch_id": branch.id,
                "file_id": file_data.file_id
            },
            "metadata": {
                "file_path": file_data.file_path,
                "language": file_data.file_extension.lstrip('.') if file_data.file_extension else "unknown",
                "content_hash": file_data.sha or "",
                "processed_at": datetime.utcnow().isoformat(),
                "version": "1.0",
                "file_size": file_data.file_size or 0,
                "encoding": "utf-8"
            },
            "nodes": [{
                "id": f"{repository.user_id}/{repository.name}/{branch.branch_name}/{file_data.file_id}/node_1",
                "type": "module",
                "name": f"{file_data.file_id}_module",
                "code_snippet": "",
                "position": {"start_line": 1, "start_column": 0, "end_line": 1, "end_column": 0},
                "properties": {"exported": False, "scope": "global"},
                "vector_data": {"semantic_text": f"Main module for {file_data.file_id}"},
                "graph_labels": ["AST_Node", "Module", "Unknown"]
            }],
            "relationships": [],
            "analysis": {
                "metrics": {
                    "line_count": 0,
                    "function_count": 0,
                    "class_count": 0,
                    "import_count": 0
                },
                "dependencies": {
                    "internal_calls": [],
                    "external_imports": []
                }
            }
        }
    except Exception as e:
        logger.error(f"Error creating empty AST JSON: {str(e)}")
        return {}

async def _store_ast_json(ast_json: dict, file_data) -> dict:
    """Store AST JSON in storage service and return storage path."""
    try:
        # Convert to JSON string
        json_content = json.dumps(ast_json, indent=2, ensure_ascii=False)
        
        # Convert to binary stream for S3 upload
        file_bytes = BytesIO(json_content.encode("utf-8"))

        # Generate storage key/path
        storage_path = f"ast/repository_id__{file_data.repository_id}/branch_id__{file_data.branch_id}/file_id__{file_data.file_id}/ast__{file_data.file_id}.json"

        # Upload to S3-compatible object storage
        upload_result = await storage_service.upload_file(
            filename=storage_path,
            archive_data=file_bytes,
            content_type="application/json",
            metadata={
                "file_id": str(file_data.file_id),
                "file_name": file_data.file_name,
                "uploaded_by": str(file_data.user_id),
            },
        )

        # Logging for debug
        logger.debug(f"💾 AST JSON stored at: {upload_result['normal_url']}")

        return upload_result
        
    except Exception as e:
        logger.error(f"Error storing AST JSON: {str(e)}")
        raise
