# app/services/kafka/kafka_push_consumer.py
import asyncio
from app.core.logger import logger
from app.db.postgress_db import get_async_db
from app.services.postgress_db_service import pg_db_service as database_service
from app.schemas.github_schemas import WebhookStatus

async def process_push_message(message_data: dict):
    """Process push messages from Kafka - handles push webhooks"""
    try:
        # logger.info(f"📤 Processing push message: {message_data}")
        
        webhook_id = message_data.get('webhook_id')
        if not webhook_id:
            logger.error("❌ No webhook_id in push message")
            return
            
        async for db in get_async_db():
            try:
                # Process push sync
                await _handle_push_webhook(webhook_id, db)
                logger.info(f"✅ Push processed for webhook {webhook_id}")
                
            except Exception as e:
                logger.error(f"❌ Error processing push for webhook {webhook_id}: {str(e)}")
                # Update webhook status to failed
                await database_service.update_webhook_status(
                    db, webhook_id, WebhookStatus.FAILED, str(e)
                )
        
    except Exception as e:
        logger.error(f"❌ Fatal error processing push message: {str(e)}")

async def _handle_push_webhook(webhook_id: int, db):
    """Handle push webhook - only uses webhook_id."""
    from app.services.github.push_sync_pipeline import PushSyncPipeline
    
    logger.info(f"📤 Processing push webhook: {webhook_id}")
    pipeline = PushSyncPipeline()
    await pipeline.sync_push(webhook_id, db)