# app/services/kafka/kafka_sync_consumer.py
import asyncio
from app.core.logger import logger
from app.db.postgress_db import get_async_db
from app.services.postgress_db_service import pg_db_service as database_service
from datetime import datetime
from fastapi.encoders import jsonable_encoder
from app.schemas.github_schemas import WebhookStatus


async def process_sync_message(key,message_data: dict):
    """Process sync messages from Kafka - handles installation webhooks"""
    try:
        logger.info(f"🔄 Processing sync message {key}: ")
        
        webhook_id = message_data.get('webhook_id')
        if not webhook_id:
            logger.error("❌ No webhook_id in sync message")
            return
            
        async for db in get_async_db():
            try:
                # Process installation sync using the webhook logic
                await _handle_installation_webhook(webhook_id, db, message_data)
                logger.info(f"✅ Sync processed for webhook {webhook_id}")
                
            except Exception as e:
                logger.error(f"❌ Error processing sync for webhook {webhook_id}: {str(e)}")
                # Update webhook status to failed
                await database_service.update_webhook_status(
                    db, webhook_id, WebhookStatus.FAILED, str(e)
                )
        
    except Exception as e:
        logger.error(f"❌ Fatal error processing sync message: {str(e)}")

async def _handle_installation_webhook(webhook_id: int, db, message_data: dict):
    """Handle installation webhook - handles all installation actions."""
    from app.services.github.installation_sync_pipeline import InstallationSyncPipeline

    # Get webhook data from database
    webhook = await database_service.get_webhook_by_id(db, webhook_id=webhook_id)
    if not webhook:
        logger.error(f"Webhook not found: {webhook_id}")
        return
        
    webhook_data = jsonable_encoder(webhook)
    payload = webhook_data["payload"]
    action = payload["action"]
    
    if action in ["created", "added", "sync_manual"]:
        # Handle installation created - sync everything
        logger.info(f"🔄 Starting installation sync for webhook {webhook_id}, action: {action}")
        pipeline = InstallationSyncPipeline()
        await pipeline.sync_installation(webhook_data["user_id"], webhook_id, db)
        
    elif action in ["deleted", "suspend"]:
        # Handle installation removed - deactivate account
        logger.info(f"🗑️ Handling installation removal for webhook {webhook_id}")
        await _handle_installation_removed(webhook_data, db)
        
    elif action == "new_permissions_accepted":
        # Handle permissions update
        logger.info(f"🔐 Handling permissions update for webhook {webhook_id}")
        await _handle_permissions_updated(webhook_data, db)
        
    else:
        logger.info(f"ℹ️ Unhandled installation action: {action} for webhook {webhook_id}")
        await database_service.update_webhook_status(
            db, webhook_id, WebhookStatus.PROCESSED
        )

async def _handle_installation_removed(webhook_data: dict, db):
    """Handle installation removal - deactivate GitHub account."""
    try:
        installation_id = webhook_data["installation_id"]
        payload = webhook_data["payload"]
        account = payload.get("installation", {}).get("account", {})
        username = account.get("login", "unknown")
        
        await database_service.update_github_account(
            db, installation_id, {
                "is_active": False,
                "suspended_at": datetime.utcnow()
            }
        )
        logger.info(f"✅ Deactivated GitHub account for {username} (installation: {installation_id})")
        
    except Exception as e:
        logger.error(f"❌ Error handling installation removal: {str(e)}")
        raise

async def _handle_permissions_updated(webhook_data: dict, db):
    """Handle installation permissions update."""
    try:
        installation_id = webhook_data["installation_id"]
        payload = webhook_data["payload"]
        installation = payload.get("installation", {})
        
        await database_service.update_github_account(
            db, installation_id, {
                "permissions": installation.get("permissions"),
                "events": installation.get("events", []),
                "updated_at": datetime.utcnow()
            }
        )
        logger.info(f"✅ Updated permissions for installation {installation_id}")
        
    except Exception as e:
        logger.error(f"❌ Error handling permissions update: {str(e)}")
        raise