# app/services/postgress_db_service.py
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import select, update, delete
from sqlalchemy.orm import selectinload
from typing import List, Optional, Dict, Any
import json
from sqlalchemy.sql import func
from app.db.postgress_db import get_async_db
from app.models.user_models import User, UserGitHubAccount
from app.models.github_models import UserRepository, UserBranch, Webhook,UserBranchFile
from app.models.sync_models import SyncJob, CodeChange
from app.core.logger import logger
from datetime import datetime

class DatabaseService:
    def __init__(self):
        self.db = get_async_db

    # ===== USER METHODS =====
    async def create_user(self, session: AsyncSession, user_data: Dict[str, Any]) -> User:
        user = User(**user_data)
        session.add(user)
        await session.commit()
        await session.refresh(user)
        return user
    async def update_user(self, session: AsyncSession, user_id: int, update_data: Dict[str, Any]) -> Optional[User]:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        if user:
            for key, value in update_data.items():
                setattr(user, key, value)
            await session.commit()
            await session.refresh(user)
        return user
    async def get_user_by_email(self, session: AsyncSession, email: str) -> Optional[User]:
        result = await session.execute(
            select(User).where(User.email == email)
        )
        return result.scalar_one_or_none()

    async def get_user_by_id(self, session: AsyncSession, user_id: int) -> Optional[User]:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        return result.scalar_one_or_none()
    async def get_user_by_installation_id(self, session: AsyncSession, installation_id: int) -> Optional[User]:
        result = await session.execute(
            select(User)
            .join(UserGitHubAccount)
            .where(UserGitHubAccount.installation_id == installation_id)
            .options(selectinload(User.github_accounts))
        )
        return result.scalar_one_or_none()
    # ===== GITHUB INSTALLATION METHODS =====
    async def get_installation_id_by_repo_and_user(
        self, session: AsyncSession, repo_id: int, user_id: int
    ) -> Optional[int]:
        """
        Returns the GitHub installation_id for a given repository and user.
        """
        result = await session.execute(
            select(UserGitHubAccount.installation_id)
            .join(UserRepository, UserRepository.github_account_id == UserGitHubAccount.id)
            .join(User, UserGitHubAccount.user_id == User.id)
            .where(
                UserRepository.repo_id == repo_id,
                User.id == user_id,
                UserGitHubAccount.is_active==True
            )
        )
        installation_id = result.scalar_one_or_none()
        return installation_id



    # ===== GITHUB ACCOUNT METHODS =====
    async def get_user_by_github_id(self, session: AsyncSession, github_id: int) -> Optional[User]:
        """Get user by GitHub ID"""
        result = await session.execute(
            select(User)
            .where(User.github_id == github_id)
        )
        return result.scalar_one_or_none()

    async def get_github_account_by_user_and_installation(self, session: AsyncSession, user_id: int, installation_id: int) -> Optional[UserGitHubAccount]:
        """Get GitHub account by user ID and installation ID"""
        result = await session.execute(
            select(UserGitHubAccount)
            .where(
                UserGitHubAccount.user_id == user_id,
                UserGitHubAccount.installation_id == installation_id
            )
        )
        return result.scalar_one_or_none()

    async def update_github_account_status(self, session: AsyncSession, account_id: int, is_active: bool) -> Optional[UserGitHubAccount]:
        """Update GitHub account active status"""
        result = await session.execute(
            select(UserGitHubAccount)
            .where(UserGitHubAccount.id == account_id)
        )
        account = result.scalar_one_or_none()
        
        if account:
            account.is_active = is_active
            account.updated_at = datetime.utcnow()
            await session.commit()
            await session.refresh(account)
        
        return account

    async def update_github_account(self, session: AsyncSession, account_id: int, update_data: Dict[str, Any]) -> Optional[UserGitHubAccount]:
        """Update GitHub account"""
        result = await session.execute(
            select(UserGitHubAccount)
            .where(UserGitHubAccount.id == account_id)
        )
        account = result.scalar_one_or_none()
        
        if account:
            for key, value in update_data.items():
                if hasattr(account, key):
                    setattr(account, key, value)
            
            account.updated_at = datetime.utcnow()
            await session.commit()
            await session.refresh(account)
        
        return account

    async def create_github_account(self, session: AsyncSession, account_data: Dict[str, Any]) -> UserGitHubAccount:
        """Create GitHub account"""
        account = UserGitHubAccount(**account_data)
        session.add(account)
        await session.commit()
        await session.refresh(account)
        return account

    async def get_github_account_by_installation_id(self, session: AsyncSession, installation_id: int) -> Optional[UserGitHubAccount]:
        """Get GitHub account by installation ID"""
        result = await session.execute(
            select(UserGitHubAccount)
            .where(UserGitHubAccount.installation_id == installation_id)
            .options(selectinload(UserGitHubAccount.user))
        )
        return result.scalar_one_or_none()
    # async def get_github_account_by_username(self, session: AsyncSession, github_username: str) -> Optional[UserGitHubAccount]:
    #     result = await session.execute(
    #         select(UserGitHubAccount)
    #         .where(UserGitHubAccount.github_username == github_username)
    #         .options(selectinload(UserGitHubAccount.user))
    #     )
    #     return result.scalar_one_or_none()

    async def update_github_account(self, session: AsyncSession, installation_id: int, update_data: Dict[str, Any]) -> Optional[UserGitHubAccount]:
        result = await session.execute(
            select(UserGitHubAccount).where(UserGitHubAccount.installation_id == installation_id)
        )
        account = result.scalar_one_or_none()
        if account:
            for key, value in update_data.items():
                setattr(account, key, value)
            await session.commit()
            await session.refresh(account)
        return account

    # ===== REPOSITORY METHODS =====
    async def create_repository(self, session: AsyncSession, repo_data: Dict[str, Any]) -> UserRepository:
        repo = UserRepository(**repo_data)
        session.add(repo)
        await session.commit()
        await session.refresh(repo)
        return repo

    async def get_repositories_by_account(self, session: AsyncSession, github_account_id: int) -> List[UserRepository]:
        result = await session.execute(
            select(UserRepository)
            .where(UserRepository.github_account_id == github_account_id)
            .options(selectinload(UserRepository.branches))
        )
        return result.scalars().all()

    # async def get_repository_by_id(self, session: AsyncSession, github_repo_id: int,github_account_id:int) -> Optional[UserRepository]:
    #     result = await session.execute(
    #         select(UserRepository)
    #         .where(UserRepository.repo_id == github_repo_id,UserRepository.github_account_id==github_account_id)
    #         .options(selectinload(UserRepository.github_account))
    #     )
    #     return result.scalar_one_or_none()
    async def get_repository_by_id(self, session: AsyncSession ,id:int) -> Optional[UserRepository]:
        result = await session.execute(
            select(UserRepository)
            .where(UserRepository.id == id)
            .options(selectinload(UserRepository.github_account))
        )
        return result.scalar_one_or_none()
    
    async def get_repository_by_archea_repo_id(self, session: AsyncSession, id:int) -> Optional[UserRepository]:
        result = await session.execute(
            select(UserRepository)
            .where(UserRepository.id == id)
            .options(selectinload(UserRepository.github_account))
        )
        return result.scalar_one_or_none()
    
    async def get_repository_by_id_and_user_id(self, session: AsyncSession, github_repo_id: int, user_id: int,installation_id:int) -> Optional[UserRepository]:
        result = await session.execute(
            select(UserRepository)
            .join(UserGitHubAccount, UserRepository.github_account_id == UserGitHubAccount.id)
            .join(User, UserGitHubAccount.user_id == User.id)
            .where(
                UserRepository.repo_id == github_repo_id,
                User.id == user_id,
                UserGitHubAccount.installation_id==installation_id
            )
            .options(selectinload(UserRepository.github_account))
        )
        return result.scalar_one_or_none()
    
    async def get_repositories_by_user_id(self, session: AsyncSession, user_id: int) -> List[UserRepository]:
        result = await session.execute(
            select(UserRepository)
            .join(UserGitHubAccount, UserRepository.github_account_id == UserGitHubAccount.id)
            .join(User, UserGitHubAccount.user_id == User.id)
            .where(User.id == user_id)
            .options(
                selectinload(UserRepository.github_account),
                selectinload(UserRepository.branches)
            )
        )
        return result.scalars().all()



    # ===== BRANCH METHODS =====
    async def create_branch(self, session: AsyncSession, branch_data: Dict[str, Any]) -> UserBranch:
        branch = UserBranch(**branch_data)
        session.add(branch)
        await session.commit()
        await session.refresh(branch)
        return branch

    async def get_branches_by_repository(self, session: AsyncSession, repository_id: int) -> List[UserBranch]:
        result = await session.execute(
            select(UserBranch).where(UserBranch.repository_id == repository_id)
        )
        return result.scalars().all()
    # ===== GITHUB ACCOUNT METHODS =====
    async def get_github_accounts_by_user_id(self, session: AsyncSession, user_id: int) -> List[UserGitHubAccount]:
        result = await session.execute(
            select(UserGitHubAccount)
            .where(UserGitHubAccount.user_id == user_id)
        )
        return result.scalars().all()
    
    async def get_branch_by_id(
        self,
        session: AsyncSession,
        id: int,
    ) -> UserBranch:
        """
        Get a branch ID by branch name, filtered by id.
        """
        query = (
            select(UserBranch)
        )
        query = query.where(
            UserBranch.id==id,
        )
    

        result = await session.execute(query)
        return result.scalar_one_or_none()
    
    async def update_branch(
        self, 
        session: AsyncSession, 
        branch_id: int, 
        update_data: Dict[str, Any]
    ) -> Optional[UserBranch]:
        """Update branch information."""
        try:
            result = await session.execute(
                select(UserBranch).where(UserBranch.id == branch_id)
            )
            branch = result.scalar_one_or_none()
            if branch:
                for key, value in update_data.items():
                    setattr(branch, key, value)
                await session.commit()
                await session.refresh(branch)
            return branch
        except Exception as e:
            logger.error(f"Error updating branch {branch_id}: {str(e)}")
            await session.rollback()
            return None

    
    async def get_branch_by_repo_id_and_name(
        self,
        session: AsyncSession,
        repo_id:int,
        branch_name: str,
    ) -> UserBranch:
        """
        Get a branch ID by branch name, filtered by user_id or installation_id.
        Priority: user_id > installation_id.
        """
        query = (
            select(UserBranch)
            .join(UserRepository, UserBranch.repository_id == UserRepository.id)
            .join(UserGitHubAccount, UserRepository.github_account_id == UserGitHubAccount.id)
        )
        query = query.where(
            UserBranch.name == branch_name,
            UserBranch.repository_id==repo_id,
        )
    

        result = await session.execute(query)
        return result.scalar_one_or_none()



    # ===== WEBHOOK METHODS =====
    async def create_webhook(self, session: AsyncSession, webhook_data: Dict[str, Any]) -> Webhook:
        webhook = Webhook(**webhook_data)
        session.add(webhook)
        await session.commit()
        await session.refresh(webhook)
        return webhook

    async def update_webhook_status(self, session: AsyncSession, archea_webhook_id: int, status: str,    error_str: Optional[str] = None  ) -> Optional[Webhook]:
        result = await session.execute(
            select(Webhook).where(Webhook.id == archea_webhook_id)
        )
        webhook = result.scalar_one_or_none()
        if webhook:
            webhook.status = status
            webhook.error_log=str(error_str)
            webhook.processed_at = func.now()
            await session.commit()
            await session.refresh(webhook)
        return webhook
    async def update_webhook(self, session: AsyncSession, archea_webhook_id: int, update_data: Dict[str, Any]) -> Optional[Webhook]:
        result = await session.execute(
            select(Webhook).where(Webhook.id == archea_webhook_id)
        )
        webhook = result.scalar_one_or_none()
        if webhook:
            for key, value in update_data.items():
                setattr(webhook, key, value)
            await session.commit()
            await session.refresh(webhook)
        return webhook
    async def update_webhook_branch(
        self,
        session: AsyncSession,
        archea_webhook_id: int,
        branch_id: int
    ) -> Optional[Webhook]:
        """
        Update the branch_id of a webhook and automatically set the repository_id
        based on the branch’s repository.
        """
        # Fetch webhook
        result = await session.execute(
            select(Webhook).where(Webhook.id == archea_webhook_id)
        )
        webhook = result.scalar_one_or_none()
        if not webhook:
            return None

        # Fetch branch to get its repository_id
        branch_result = await session.execute(
            select(UserBranch).where(UserBranch.id == branch_id)
        )
        branch = branch_result.scalar_one_or_none()
        if not branch:
            return None

        # Update both branch_id and repository_id
        webhook.branch_id = branch_id
        webhook.repository_id = branch.repository_id

        await session.commit()
        await session.refresh(webhook)
        return webhook



    # ===== WEBHOOK METHODS =====
    async def get_webhook_by_id(
        self, 
        session: AsyncSession, 
        webhook_id: int
    ) -> Optional[Webhook]:
        """
        Fetch a specific webhook by its ID with all relationships loaded.
        """
        try:
            result = await session.execute(
                select(Webhook)
                .where(Webhook.id == webhook_id)
                .options(
                    selectinload(Webhook.repository),
                    selectinload(Webhook.branch)
                )
            )
            webhook = result.scalar_one_or_none()
            
            if not webhook:
                logger.warning(f"Webhook not found with ID: {webhook_id}")
                return None
                
            logger.debug(f"Retrieved webhook {webhook_id}")
            return webhook
            
        except Exception as e:
            logger.error(f"Error fetching webhook {webhook_id}: {str(e)}")
            return None

    # Also add this method if it's missing:
    async def get_webhook_by_delivery_id(
        self, 
        session: AsyncSession, 
        delivery_id: str
    ) -> Optional[Webhook]:
        """
        Fetch a webhook by its GitHub delivery ID.
        """
        try:
            result = await session.execute(
                select(Webhook)
                .where(Webhook.delivery_id == delivery_id)
                .options(
                    selectinload(Webhook.repository),
                    selectinload(Webhook.branch)
                )
            )
            webhook = result.scalar_one_or_none()
            return webhook
            
        except Exception as e:
            logger.error(f"Error fetching webhook by delivery ID {delivery_id}: {str(e)}")
            return None

    # FIX: This method was incorrectly indented inside the previous method
    async def get_webhooks_by_user_id(
        self, 
        session: AsyncSession, 
        user_id: int, 
        limit: int = 50
    ) -> List[Webhook]:
        """
        Fetch recent webhooks belonging to a given user, limited to the most recent `limit` entries.
        """
        result = await session.execute(
            select(Webhook)
            .where(Webhook.user_id == user_id)
            .order_by(Webhook.created_at.desc())
            .limit(limit)
            .options(selectinload(Webhook.repository))  # Optional: eager load related repo
        )
        return result.scalars().all()

    # ===== SYNC JOB METHODS =====
    async def create_sync_job(self, session: AsyncSession, job_data: Dict[str, Any]) -> SyncJob:
        job = SyncJob(**job_data)
        session.add(job)
        await session.commit()
        await session.refresh(job)
        return job

    async def update_sync_job_status(self, session: AsyncSession, job_id: int, status: str, error_message: str = None) -> Optional[SyncJob]:
        result = await session.execute(
            select(SyncJob).where(SyncJob.id == job_id)
        )
        job = result.scalar_one_or_none()
        if job:
            job.status = status
            if status == "running":
                job.started_at = func.now()
            elif status in ["completed", "failed"]:
                job.completed_at = func.now()
            if error_message:
                job.error_message = error_message
            await session.commit()
            await session.refresh(job)
        return job
    # ===== SYNC JOB METHODS =====
    async def get_sync_job_by_id(self, session: AsyncSession, job_id: int) -> Optional[SyncJob]:
        """
        Fetch a SyncJob by its ID.
        """
        result = await session.execute(
            select(SyncJob).where(SyncJob.id == job_id)
        )
        return result.scalar_one_or_none()

    # ===== CODE CHANGE METHODS =====
    async def create_code_change(self, session: AsyncSession, change_data: Dict[str, Any]) -> CodeChange:
        change = CodeChange(**change_data)
        session.add(change)
        await session.commit()
        await session.refresh(change)
        return change

    async def get_code_changes_by_user(self, session: AsyncSession, user_id: int, limit: int = 50) -> List[CodeChange]:
        result = await session.execute(
            select(CodeChange)
            .join(SyncJob)
            .where(SyncJob.user_id == user_id)
            .order_by(CodeChange.created_at.desc())
            .limit(limit)
        )
        return result.scalars().all()
    # ===== USER BRANCH FILE METHODS =====
    async def create_file(self, session: AsyncSession, file_data: Dict[str, Any]) -> UserBranchFile:
        file = UserBranchFile(**file_data)
        session.add(file)
        await session.commit()
        await session.refresh(file)
        return file
    async def update_file(self, session: AsyncSession, id: int, file_data: Dict[str, Any]) -> UserBranchFile:
        """Update existing file."""
        # First get the file
        file = await session.get(UserBranchFile, id)
        if not file:
            raise ValueError(f"File with ID {id} not found")
        
        # Update fields
        for key, value in file_data.items():
            if hasattr(file, key):
                setattr(file, key, value)
        
        file.updated_at = datetime.utcnow()
        
        await session.commit()
        await session.refresh(file)
        return file


    async def get_file_by_file_id(self, session: AsyncSession, file_id: str) -> Optional[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile)
            .where(UserBranchFile.file_id == file_id)
            .options(
                selectinload(UserBranchFile.repository),
                selectinload(UserBranchFile.branch)
            )
        )
        return result.scalar_one_or_none()
    async def get_file_by__id(self, session: AsyncSession, id: int) -> Optional[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile)
            .where(UserBranchFile.id == id)
            .options(
                selectinload(UserBranchFile.repository),
                selectinload(UserBranchFile.branch)
            )
        )
        return result.scalar_one_or_none()

    async def get_files_by_branch_id(self, session: AsyncSession, branch_id: int) -> List[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile)
            .where(UserBranchFile.branch_id == branch_id)
            .options(selectinload(UserBranchFile.repository))
        )
        return result.scalars().all()

    async def get_files_by_repository_id(self, session: AsyncSession, repository_id: int) -> List[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile)
            .where(UserBranchFile.repository_id == repository_id)
            .options(selectinload(UserBranchFile.branch))
        )
        return result.scalars().all()

    async def update_file_status(self, session: AsyncSession, file_id: str, status: str) -> Optional[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile).where(UserBranchFile.file_id == file_id)
        )
        file = result.scalar_one_or_none()
        if file:
            file.file_status = status
            file.updated_at = func.now()
            await session.commit()
            await session.refresh(file)
        return file

    async def mark_file_inactive(self, session: AsyncSession, file_id: str) -> Optional[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile).where(UserBranchFile.file_id == file_id)
        )
        file = result.scalar_one_or_none()
        if file:
            file.is_active = False
            file.file_status = "deleted"
            file.updated_at = func.now()
            await session.commit()
            await session.refresh(file)
        return file

    async def get_active_files_by_branch(self, session: AsyncSession, branch_id: int) -> List[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile)
            .where(
                UserBranchFile.branch_id == branch_id,
                UserBranchFile.is_active == True
            )
            .options(selectinload(UserBranchFile.repository))
        )
        return result.scalars().all()

    async def get_file_by_path_and_branch(self, session: AsyncSession, file_path: str, branch_id: int) -> Optional[UserBranchFile]:
        result = await session.execute(
            select(UserBranchFile)
            .where(
                UserBranchFile.file_path == file_path,
                UserBranchFile.branch_id == branch_id
            )
        )
        return result.scalar_one_or_none()

# Create service instance
pg_db_service = DatabaseService()