import os
import uuid
import shutil
import json
import tempfile
from typing import Optional, Dict, Any, BinaryIO
from datetime import datetime, timedelta
from fastapi import HTTPException
import boto3
from botocore.exceptions import ClientError, NoCredentialsError
import io
from app.core.config import settings
from app.core.logger import logger

class StorageService:
    def __init__(self):
        self.storage_type = settings.STORAGE_TYPE  # 'local' or 's3'
        self.local_storage_path = settings.LOCAL_STORAGE_PATH
        
        # S3 configuration
        self.s3_bucket = settings.S3_BUCKET_NAME
        self.s3_region = settings.S3_REGION
        self.s3_presigned_url_expiry = getattr(settings, 'S3_PRESIGNED_URL_EXPIRY', 3600)  # 1 hour default
        self.s3_endpoint_url = settings.S3_ENDPOINT_URL
        
        # Initialize S3 client if using S3
        if self.storage_type == 's3':
            try:
                # For Hetzner Object Storage, we need to configure the endpoint properly
                self.s3_client = boto3.client(
                    's3',
                    endpoint_url=f"https://{self.s3_endpoint_url}",  # Full endpoint URL
                    aws_access_key_id=settings.AWS_ACCESS_KEY_ID,
                    aws_secret_access_key=settings.AWS_SECRET_ACCESS_KEY,
                    region_name=self.s3_region
                )
                logger.info(f"Initialized S3 client for bucket: {self.s3_bucket}, endpoint: {self.s3_endpoint_url}")
            except NoCredentialsError:
                logger.error("AWS credentials not found")
                raise
            except Exception as e:
                logger.error(f"Failed to initialize S3 client: {str(e)}")
                raise
        else:
            # Ensure local storage directory exists
            os.makedirs(self.local_storage_path, exist_ok=True)
            logger.info(f"Initialized local storage at: {self.local_storage_path}")

    async def upload_repo_archive(
        self, 
        repo_name: str, 
        archive_data: BinaryIO, 
        content_type: str = "application/zip",
        metadata: Optional[Dict[str, str]] = None,
        custom_filename: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Upload repository archive to storage and return download URL.
        """
        if custom_filename:
            filename = custom_filename
        else:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            filename = f"{repo_name}_{timestamp}_{uuid.uuid4().hex[:8]}.zip"
        
        if self.storage_type == 's3':
            return await self._upload_to_s3(filename, archive_data, content_type, metadata)
        else:
            return await self._upload_to_local(filename, archive_data, metadata)

    async def upload_file(
        self,
        filename: str,
        archive_data: BinaryIO,
        content_type: str,
        metadata: Optional[Dict[str, str]] = None,
    ) -> Dict[str, Any]:
        """
        Upload file depending on configured storage type.
        """
        if self.storage_type == "s3":
            return await self._upload_to_s3(filename, archive_data, content_type, metadata)
        else:
            return await self._upload_to_local(filename, archive_data, metadata)

    async def _upload_to_s3(
        self,
        filename: str,
        archive_data: BinaryIO,
        content_type: str,
        metadata: Optional[Dict[str, str]] = None
    ) -> Dict[str, Any]:
        """
        Upload file to Hetzner (S3-compatible) Object Storage.
        """
        try:
            extra_args = {"ContentType": content_type}
            if metadata:
                # Convert all metadata values to strings
                string_metadata = {k: str(v) for k, v in metadata.items()}
                extra_args["Metadata"] = string_metadata

            archive_data.seek(0)

            self.s3_client.upload_fileobj(
                archive_data,
                self.s3_bucket,
                filename,
                ExtraArgs=extra_args
            )

            logger.info(f"✅ Uploaded {filename} to S3 bucket {self.s3_bucket}")

            # Generate presigned URL
            presigned_url = self._generate_s3_presigned_url(filename)

            # For Hetzner, use the correct URL format
            normal_url = f"https://{self.s3_bucket}.{self.s3_endpoint_url}/{filename}"

            return {
                "storage_type": "s3",
                "filename": filename,
                "bucket": self.s3_bucket,
                "s3_path": f"s3://{self.s3_bucket}/{filename}",
                "normal_url": normal_url,
                "download_url": presigned_url,
                "expires_in": self.s3_presigned_url_expiry,
                "uploaded_at": datetime.utcnow().isoformat()
            }

        except ClientError as e:
            logger.error(f"S3 upload error: {e}")
            raise HTTPException(status_code=500, detail=f"S3 upload failed: {str(e)}")
        except Exception as e:
            logger.error(f"Unexpected error during S3 upload: {e}")
            raise HTTPException(status_code=500, detail=f"Upload failed: {str(e)}")

    async def _upload_to_local(
        self, 
        filename: str, 
        archive_data: BinaryIO,
        metadata: Optional[Dict[str, str]] = None
    ) -> Dict[str, Any]:
        """
        Save file to local filesystem and generate local URL.
        """
        try:
            base_dir = os.path.abspath(self.local_storage_path)
            file_path = os.path.join(base_dir, filename)

            os.makedirs(os.path.dirname(file_path), exist_ok=True)
            
            archive_data.seek(0)
            
            with open(file_path, 'wb') as f:
                shutil.copyfileobj(archive_data, f)
            
            logger.info(f"Successfully saved {filename} to local storage: {file_path}")
            
            download_url = f"/storage/download/{filename}"
            
            return {
                "storage_type": "local",
                "filename": filename,
                "file_path": file_path,
                "normal_url": download_url,
                "download_url": download_url,
                "uploaded_at": datetime.now().isoformat(),
                "local_path": file_path
            }
            
        except Exception as e:
            logger.error(f"Local storage upload error: {e}")
            raise HTTPException(status_code=500, detail=f"Local storage upload failed: {str(e)}")
    
    def _generate_s3_presigned_url(self, filename: str) -> str:
        """
        Generate presigned URL for S3 object download.
        """
        try:
            url = self.s3_client.generate_presigned_url(
                'get_object',
                Params={
                    'Bucket': self.s3_bucket,
                    'Key': filename
                },
                ExpiresIn=self.s3_presigned_url_expiry
            )
            return url
        except ClientError as e:
            logger.error(f"Error generating presigned URL: {e}")
            raise HTTPException(status_code=500, detail="Failed to generate download URL")

    async def download_file(self, filename: str) -> Dict[str, Any]:
        """
        Get download information for a file.
        """
        if self.storage_type == 's3':
            return await self._get_s3_download_info(filename)
        else:
            return await self._get_local_download_info(filename)

    async def _get_s3_download_info(self, filename: str) -> Dict[str, Any]:
        """
        Get S3 file download information with fresh presigned URL.
        """
        try:
            # Check if file exists
            self.s3_client.head_object(Bucket=self.s3_bucket, Key=filename)
            
            # Generate fresh presigned URL
            download_url = self._generate_s3_presigned_url(filename)
            
            return {
                "storage_type": "s3",
                "filename": filename,
                "bucket": self.s3_bucket,
                "download_url": download_url,
                "expires_in": self.s3_presigned_url_expiry,
                "available": True
            }
            
        except ClientError as e:
            if e.response['Error']['Code'] == '404':
                raise HTTPException(status_code=404, detail="File not found")
            else:
                logger.error(f"S3 error: {e}")
                raise HTTPException(status_code=500, detail=f"S3 error: {str(e)}")

    async def _get_local_download_info(self, filename: str) -> Dict[str, Any]:
        """
        Get local file download information.
        """
        file_path = os.path.join(self.local_storage_path, filename)
        
        if not os.path.exists(file_path):
            raise HTTPException(status_code=404, detail="File not found")
        
        return {
            "storage_type": "local",
            "filename": filename,
            "file_path": file_path,
            "download_url": f"/storage/download/{filename}",
            "file_size": os.path.getsize(file_path),
            "available": True
        }

    async def delete_file(self, filename: str) -> bool:
        """
        Delete a file from storage.
        """
        try:
            if self.storage_type == 's3':
                self.s3_client.delete_object(Bucket=self.s3_bucket, Key=filename)
                logger.info(f"Deleted file from S3: {filename}")
            else:
                file_path = os.path.join(self.local_storage_path, filename)
                if os.path.exists(file_path):
                    os.remove(file_path)
                    logger.info(f"Deleted file from local storage: {filename}")
                else:
                    raise HTTPException(status_code=404, detail="File not found")
            
            return True
            
        except ClientError as e:
            logger.error(f"S3 delete error: {e}")
            raise HTTPException(status_code=500, detail=f"S3 delete failed: {str(e)}")
        except Exception as e:
            logger.error(f"Delete error: {e}")
            raise HTTPException(status_code=500, detail=f"Delete failed: {str(e)}")

    # Additional method for AST JSON storage
    async def upload_ast_json(
        self,
        repo_id: int,
        branch_id: int,
        file_id: str,
        ast_data: Dict[str, Any],
        metadata: Optional[Dict[str, str]] = None
    ) -> Dict[str, Any]:
        """
        Upload AST JSON data to storage.
        """
        try:
            # Create structured filename
            filename = f"ast/{repo_id}/{branch_id}/{file_id}/ast_{file_id}.json"
            
            # Convert AST data to JSON bytes
            json_bytes = json.dumps(ast_data, indent=2).encode('utf-8')
            json_io = io.BytesIO(json_bytes)
            
            # Prepare metadata
            if metadata is None:
                metadata = {}
            
            metadata.update({
                "repo_id": str(repo_id),
                "branch_id": str(branch_id),
                "file_id": file_id,
                "content_type": "application/json",
                "uploaded_at": datetime.utcnow().isoformat()
            })
            
            # Upload to storage
            result = await self.upload_file(
                filename=filename,
                archive_data=json_io,
                content_type="application/json",
                metadata=metadata
            )
            
            logger.info(f"✅ Uploaded AST JSON for file {file_id} to {filename}")
            return result
            
        except Exception as e:
            logger.error(f"Failed to upload AST JSON for file {file_id}: {str(e)}")
            raise

    async def download_ast_json(
        self,
        repo_id: int,
        branch_id: int,
        file_id: str
    ) -> Optional[Dict[str, Any]]:
        """
        Download and parse AST JSON data from storage.
        """
        try:
            filename = f"ast/{repo_id}/{branch_id}/{file_id}/ast_{file_id}.json"
            
            if self.storage_type == 's3':
                # Get the file from S3
                response = self.s3_client.get_object(Bucket=self.s3_bucket, Key=filename)
                content = response['Body'].read().decode('utf-8')
                return json.loads(content)
            else:
                # Get from local storage
                file_path = os.path.join(self.local_storage_path, filename)
                if os.path.exists(file_path):
                    with open(file_path, 'r') as f:
                        return json.load(f)
                return None
                
        except ClientError as e:
            if e.response['Error']['Code'] == 'NoSuchKey':
                logger.warning(f"AST JSON not found for file {file_id}")
                return None
            else:
                logger.error(f"Error downloading AST JSON for file {file_id}: {str(e)}")
                raise
        except Exception as e:
            logger.error(f"Unexpected error downloading AST JSON for file {file_id}: {str(e)}")
            return None

# Create service instance
storage_service = StorageService()