# app/services/webhook/background_processor.py
from app.core.logger import logger
from app.services.webhook.webhook_manager import WebhookManager
from app.services.postgress_db_service import pg_db_service as database_service
from datetime import datetime, timedelta
import asyncio
from app.core.config import settings

INSTALLATION_WAIT_TIMEOUT = settings.INSTALLATION_WAIT_TIMEOUT
class WebhookBackgroundProcessor:
    """Background processor that only needs webhook_id."""
    
    def __init__(self):
        self.webhook_manager = WebhookManager()
    
    async def process_webhook(
        self,
        archea_webhook_id: int,
        db_session_context
    ):
        """Process webhook with all the business logic moved from API endpoint."""
        try:
            async for db in db_session_context():
                # Get webhook record
                webhook = await database_service.get_webhook_by_id(db, archea_webhook_id)
                if not webhook:
                    logger.error(f"Webhook not found: {archea_webhook_id}")
                    return
                
                payload = webhook.payload
                event_type = webhook.event_type
                delivery_id = webhook.delivery_id
                
                # Update status to processing
                await database_service.update_webhook_status(
                    db, archea_webhook_id, "processing", None
                )
                
                # Extract installation and sender info
                installation_data = payload.get("installation", {})
                installation_id = installation_data.get("id")
                sender_data = payload.get("sender", {})
                requester_data = payload.get("requester", {})
                sender_github_id = sender_data.get("id")
                action = payload.get("action", "unknown")
                account_data = installation_data.get("account", {})
                
                if not installation_id:
                    logger.warning(f"No installation ID in webhook payload for {delivery_id}")
                    await database_service.update_webhook_status(
                        db, archea_webhook_id, "failed", "Missing installation ID in payload"
                    )
                    return
                #update webhook with installation_id
                await database_service.update_webhook(
                    db, archea_webhook_id, {"installation_id": installation_id}
                )
                # FIRST: Check if GitHub account exists by installation ID
                start_time = datetime.utcnow()
                user = None
                github_account = None
                
                while start_time + timedelta(seconds=INSTALLATION_WAIT_TIMEOUT) > datetime.utcnow():
                    github_account = await database_service.get_github_account_by_installation_id(db, installation_id)
                    if github_account:
                        break
                    await asyncio.sleep(1)  # Wait before retrying
                
                if github_account:
                    # GitHub account found - get the associated user
                    user = github_account.user
                    logger.info(f"Found existing GitHub account for installation: {installation_id}, user: {user.id}")
                else:
                    # No GitHub account found - find user by sender's GitHub ID
                    user_github_id = None
                    if action == "created" and event_type == "installation":
                        # if requester_data means the data comes from admin approval
                        user_github_id = requester_data.get("id") if requester_data else sender_github_id
                    elif action == "deleted" and event_type == "installation":
                        user_github_id = sender_github_id
                    else:
                        user_github_id = sender_github_id
                        
                    if user_github_id is None:
                        await database_service.update_webhook_status(
                            db, archea_webhook_id, "failed", "Sender GitHub ID required in payload"
                        )
                        return
                    
                    start_time = datetime.utcnow()
                    user = await database_service.get_user_by_github_id(db, user_github_id)
                    
                    # Wait for user to appear
                    user = await database_service.get_user_by_github_id(db, user_github_id)
                    if not user:
                        await database_service.update_webhook_status(
                            db, archea_webhook_id, "failed", f"User not found for sender GitHub ID ,{user_github_id}"
                        )
                        return
                    
                    logger.info(f"Found user by sender GitHub ID: {sender_github_id}, user: {user.id}")
                    
                    # Create or update GitHub account
                    github_account = await database_service.get_github_account_by_installation_id(db, installation_id)
                    if not github_account:
                        # Create new GitHub account
                        github_account_data = {
                            "user_id": user.id,
                            "github_username": account_data.get("login"),
                            "github_id": account_data.get("id"),
                            "installation_id": installation_id,
                            "account_type": account_data.get("type", "User"),
                            "avatar_url": account_data.get("avatar_url"),
                            "html_url": account_data.get("html_url"),
                            "permissions": installation_data.get("permissions"),
                            "is_active": True,
                            "created_at": datetime.utcnow()
                        }
                        github_account = await database_service.create_github_account(db, github_account_data)
                        logger.info(f"Created new GitHub account for installation: {installation_id}")
                
                # Handle installation deletion
                if action == "deleted" and github_account:
                    # Deactivate GitHub account
                    await database_service.update_github_account(
                        db, github_account.id, {"is_active": False}
                    )
                    logger.info(f"Deactivated GitHub account for installation: {installation_id}")
                
                # Associate with repository if available
                repository = payload.get("repository", {})
                repo_id = repository.get("id")
                if repo_id:
                    repo = await database_service.get_repository_by_id_and_user_id(
                        db, repo_id, user.id, installation_id
                    )
                    if repo:
                        await database_service.update_webhook(db, archea_webhook_id, {"repository_id": repo.id})
                
                # Update webhook with user ID
                await database_service.update_webhook(db, archea_webhook_id, {"user_id": user.id})
                
                logger.info(
                    f"Webhook processing completed: {event_type} for user {user.id} "
                    f"(delivery: {delivery_id}, archea_webhook_id: {archea_webhook_id})"
                )
                # Now process the webhook using WebhookManager
                await self.webhook_manager.process_webhook(archea_webhook_id, db)
                # # Update status to completed
                await database_service.update_webhook_status(
                    db, archea_webhook_id, "completed", None
                )
                
        except Exception as e:
            logger.error(f"Background processing failed for webhook {archea_webhook_id}: {str(e)}")
            
            # Update webhook status to failed
            try:
                async for db in db_session_context():
                    await database_service.update_webhook_status(
                        db, archea_webhook_id, "failed", str(e)
                    )
            except Exception as update_error:
                logger.error(f"Failed to update webhook status: {str(update_error)}")
# Initialize the background processor
webhook_processor = WebhookBackgroundProcessor()