# app/services/webhook/webhook_manager.py
from sqlalchemy.ext.asyncio import AsyncSession

from app.core.logger import logger
from app.services.postgress_db_service import pg_db_service as database_service
from datetime import datetime
from fastapi.encoders import jsonable_encoder
from app.services.kafka.kafka_client import send_ast_processing_message, send_chunking_message, send_push_message, send_webhook_message, send_sync_message

from app.schemas.github_schemas import WebhookStatus

class WebhookManager:
    """Manages webhook processing by routing to Kafka topics."""
    
    async def process_webhook(
        self,
        archea_webhook_id: int,
        db: AsyncSession
    ) -> None:
        """Process webhook - only needs webhook_id."""
        try:
            logger.info(f"Processing webhook: {archea_webhook_id}")
            
            # Step 1: Get webhook data
            webhook_data = await self._get_webhook_data(archea_webhook_id, db)
            if not webhook_data:
                return
            
            # Step 2: Mark as processing
            await self._update_webhook_status(archea_webhook_id, WebhookStatus.PROCESSING, db)
            
            # Step 3: Route based on event type and send to Kafka
            event_type = webhook_data["event_type"]
            # user_id = webhook_data["user_id"]
            
            if event_type == "installation":
                # Send to Kafka sync topic
                await self._send_installation_to_kafka(archea_webhook_id, webhook_data)
            elif event_type == "push":
                # Send to Kafka push topic
                await self._send_push_to_kafka(archea_webhook_id, webhook_data)
            else:
                # Skip as of now - mark completed
                await self._handle_other_webhook(archea_webhook_id, db)
            
            # Step 4: Mark as completed
            await self._update_webhook_status(archea_webhook_id, WebhookStatus.PROCESSED, db)
            logger.info(f"Webhook processed successfully: {archea_webhook_id}")
            
        except Exception as e:
            logger.error(f"Webhook processing failed {archea_webhook_id}: {str(e)}")
            await self._update_webhook_status(archea_webhook_id, WebhookStatus.FAILED, db, str(e))

    async def _handle_other_webhook(
        self,
        webhook_id: int,
        db: AsyncSession
    ) -> None:
        """Handle other webhook types - only uses webhook_id."""
        logger.info(f"Processing other webhook type: {webhook_id}")
        # Mark as processed for now
        await self._update_webhook_status(webhook_id, WebhookStatus.PROCESSED, db)
    
    async def _send_installation_to_kafka(self, webhook_id: int, webhook_data: dict):
        """Send installation webhook to Kafka sync topic"""
        try:
            kafka_message = {
                "webhook_id": webhook_id,
                "event_type": "installation",
                "user_id": webhook_data["user_id"],
                "installation_id": webhook_data["installation_id"],
                "payload": webhook_data["payload"],
                "timestamp": datetime.utcnow().isoformat(),
                "source": "webhook_manager"
            }
            
            success = await send_sync_message(
                f"user_{webhook_data['user_id']}",
                kafka_message
            )
            
            if success:
                logger.info(f"✅ Installation webhook {webhook_id} sent to Kafka sync topic")
            else:
                logger.error(f"❌ Failed to send installation webhook {webhook_id} to Kafka")
                raise Exception("Failed to send to Kafka")
                
        except Exception as e:
            logger.error(f"Error sending installation to Kafka: {str(e)}")
            raise
    
    async def _send_push_to_kafka(self, webhook_id: int, webhook_data: dict):
        """Send push webhook to Kafka push topic"""
        try:
            kafka_message = {
                "webhook_id": webhook_id,
                "event_type": "push",
                "user_id": webhook_data["user_id"],
                "installation_id": webhook_data["installation_id"],
                "repository_id": webhook_data.get("repository_id"),
                "payload": webhook_data["payload"],
                "timestamp": datetime.utcnow().isoformat(),
                "source": "webhook_manager"
            }
            
            repo_id = webhook_data.get("repository_id", "unknown")
            success = await send_push_message(
                f"repo_{repo_id}",
                kafka_message
            )
            
            if success:
                logger.info(f"✅ Push webhook {webhook_id} sent to Kafka push topic")
            else:
                logger.error(f"❌ Failed to send push webhook {webhook_id} to Kafka")
                raise Exception("Failed to send to Kafka")
                
        except Exception as e:
            logger.error(f"Error sending push to Kafka: {str(e)}")
            raise
    
    async def _get_webhook_data(
        self,
        webhook_id: int,
        db: AsyncSession
    ) -> dict:
        """Get webhook data - only uses webhook_id."""
        webhook = await database_service.get_webhook_by_id(db, webhook_id)
        if not webhook:
            raise ValueError(f"Webhook not found: {webhook_id}")
        
        return {
            "event_type": webhook.event_type,
            "payload": webhook.payload,
            "user_id": webhook.user_id,
            "installation_id": webhook.installation_id,
            "repository_id": webhook.repository_id
        }
    
    async def _update_webhook_status(
        self,
        webhook_id: int,
        status: str,
        db: AsyncSession,
        error: str = None
    ) -> None:
        """Update webhook status - only uses webhook_id."""
        await database_service.update_webhook_status(
            db, webhook_id, status, error
        )
    
    # async def _handle_other_webhook(
    #     self,
    #     webhook_id: int,
    #     db: AsyncSession
    # ) -> None:
    #     """Handle other webhook types - only uses webhook_id."""
    #     logger.info(f"Processing other webhook type: {webhook_id}")
    #     # Mark as processed for now
    #     await self._update_webhook_status(webhook_id, WebhookStatus.PROCESSED, db)