# app/routers/github_router.py
from fastapi import APIRouter, Request, HTTPException, Query, Depends, BackgroundTasks, status
from fastapi.responses import RedirectResponse, JSONResponse
from sqlalchemy.ext.asyncio import AsyncSession
import json
import asyncio
import uuid
from datetime import datetime, timedelta
from typing import Optional

from app.db.postgress_db import get_async_db
from app.services.postgress_db_service import pg_db_service as database_service
from app.core.config import settings
from app.core.logger import logger
from app.services.github_service import github_service
from app.schemas.github_schemas import (
    # Request schemas
    GitHubInstallRequest,
    TestWebhookRequest,
    ManualSyncRequest,
    
    # Response schemas
    RepositoryListConfigResponse,
    GitHubCallbackResponse,
    BranchListConfResponse,
    WebhookTestResponse,
    ManualSyncResponse,
    BranchInfo,
    RepositoryInfo,
    # Enums
    WebhookStatus,
    SyncJobStatus
)
from app.schemas.error_schemas import ErrorResponse

github_router = APIRouter()

# Constants
INSTALLATION_WAIT_TIMEOUT = settings.INSTALLATION_WAIT_TIMEOUT
WEBHOOK_PROCESSING_DELAY = settings.WEBHOOK_PROCESSING_DELAY
MAX_WEBHOOK_PAYLOAD_SIZE = settings.MAX_WEBHOOK_PAYLOAD_SIZE

@github_router.post(
    "/install",
    summary="Install GitHub App",
    response_model=GitHubCallbackResponse,
    responses={
        400: {"model": ErrorResponse},
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def github_app_install(
    request: GitHubInstallRequest,
    db: AsyncSession = Depends(get_async_db)
):
    """Redirect users to install the GitHub App with proper user validation."""
    try:
        user_id = request.user_id
        
        # Check if user exists in database
        user = await database_service.get_user_by_id(db, user_id)
        if not user:
            logger.warning(f"Installation attempt for non-existent user: {user_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found. Please ensure the user exists before installing GitHub App."
            )
        
        # Check if user already has active GitHub installation
        existing_accounts = await database_service.get_github_accounts_by_user_id(db, user_id)
        active_accounts = [acc for acc in existing_accounts if acc.is_active]
        
        if active_accounts:
            logger.info(f"User {user_id} already has active GitHub installations")
            return GitHubCallbackResponse(
                message="GitHub App already installed",
                installation_id=active_accounts[0].installation_id,
                setup_action="existing",
                username=active_accounts[0].github_username,
                account_type=active_accounts[0].account_type,
                user_id=user_id
            )
        
        # Generate installation URL
        install_url = await github_service.get_new_installation_user(user_id=user_id)
        
        logger.info(f"Generated installation URL for user: {user_id}")
        
        return GitHubCallbackResponse(
            message="GitHub App installation initiated",
            installation_id=0,  # Will be set after callback
            setup_action="install",
            username=None,
            account_type=None,
            user_id=user_id,
            install_url=install_url
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in GitHub install for user {request.user_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error during GitHub App installation"
        )

@github_router.post(
    "/webhook",
    summary="GitHub Webhook Handler",
    responses={
        400: {"model": ErrorResponse},
        403: {"model": ErrorResponse},
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def github_webhook(
    request: Request,
    background_tasks: BackgroundTasks,
    db: AsyncSession = Depends(get_async_db)
):
    """Receive GitHub App webhook events with production-grade validation."""
    webhook_id = None
    delivery_id = request.headers.get("X-GitHub-Delivery", str(uuid.uuid4()))
    
    try:
        # Read and verify webhook payload
        body_bytes = await request.body()
        
        # Check payload size
        if len(body_bytes) > MAX_WEBHOOK_PAYLOAD_SIZE:
            raise HTTPException(
                status_code=status.HTTP_413_CONTENT_TOO_LARGE,
                detail=f"Webhook payload too large. Maximum size is {MAX_WEBHOOK_PAYLOAD_SIZE} bytes"
            )
        
        signature = request.headers.get("X-Hub-Signature-256", "")
        event_type = request.headers.get("X-GitHub-Event", "unknown")
        
        # Verify webhook signature
        if not signature:
            logger.warning(f"Missing signature for webhook delivery: {delivery_id}")
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Missing webhook signature"
            )

        if not github_service.verify_webhook_signature(body_bytes, signature):
            logger.warning(f"Invalid webhook signature for delivery: {delivery_id}")
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail="Invalid webhook signature"
            )
        
        # Parse payload
        try:
            payload = json.loads(body_bytes.decode('utf-8'))
        except json.JSONDecodeError as e:
            logger.error(f"Invalid JSON in webhook payload for {delivery_id}: {str(e)}")
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Invalid JSON payload"
            )
        
        # Extract installation ID and validate
        installation_id = payload.get("installation", {}).get("id")
        if not installation_id:
            logger.warning(f"No installation ID in webhook payload for {delivery_id}")
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Installation ID required"
            )
        
        # Find user and GitHub account
        user = await database_service.get_user_by_installation_id(db, installation_id)
        if not user:
            logger.warning(f"No user found for installation ID: {installation_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found for this installation"
            )
        
        github_account = await database_service.get_github_account_by_installation_id(db, installation_id)
        if not github_account:
            logger.warning(f"No GitHub account found for installation ID: {installation_id}")
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="GitHub account not found"
            )
        
        # Store webhook in database
        webhook_data = {
            "webhook_id": int(request.headers.get("X-GitHub-Hook-ID", 0)),
            "event_type": event_type,
            "payload": payload,
            "headers": dict(request.headers),
            "delivery_id": delivery_id,
            "signature": signature[:255],  # Truncate if too long
            "user_id": user.id,
            "status": WebhookStatus.RECEIVED,
            "installation_id":installation_id,
            "created_at": datetime.utcnow()
        }
        
        # Associate with repository if available
        repository = payload.get("repository", {})
        repo_id = repository.get("id")
        if repo_id:
            repo = await database_service.get_repository_by_id(db, repo_id, github_account.id)
            if repo:
                webhook_data["repository_id"] = repo.id
        
        webhook = await database_service.create_webhook(db, webhook_data)
        archea_webhook_id = webhook.id
        
        logger.info(
            f"Webhook received: {event_type} for user {user.id} "
            f"(delivery: {delivery_id}, archea_webhook_id: {archea_webhook_id})"
        )
        
        # Process webhook in background
        background_tasks.add_task(
            process_webhook_background,
            archea_webhook_id=archea_webhook_id,
            event_type=event_type,
            payload=payload,
            user_id=user.id,
            installation_id=installation_id,
            db_session_context=get_async_db
        )
        
        return JSONResponse(
            status_code=status.HTTP_202_ACCEPTED,
            content={
                "status": "accepted",
                "event": event_type,
                "webhook_id": webhook_id,
                "user_id": user.id,
                "delivery_id": delivery_id,
                "message": "Webhook is being processed"
            }
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Unexpected error processing webhook {delivery_id}: {str(e)}")
        
        # Update webhook status if it was created
        if webhook_id:
            try:
                await database_service.update_webhook_status(
                    db, webhook_id, WebhookStatus.FAILED, str(e)
                )
            except Exception as update_error:
                logger.error(f"Failed to update webhook status: {str(update_error)}")
        
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error processing webhook"
        )

async def process_webhook_background(
    archea_webhook_id: int,
    event_type: str,
    payload: dict,
    user_id: str,
    installation_id: int,
    db_session_context
):
    """Background task to process webhook events with proper error handling."""
    try:
        async for db in db_session_context():
            # Add small delay for database consistency
            await asyncio.sleep(WEBHOOK_PROCESSING_DELAY)
            
            # Update webhook status to processing
            await database_service.update_webhook_status(
                db, archea_webhook_id, WebhookStatus.PROCESSING
            )
            
            # Route to appropriate event handler
            try:
                if event_type == "installation":
                    await _handle_installation_event(payload, db, user_id, archea_webhook_id)
                elif event_type == "push":
                    await _handle_push_event(payload, db, user_id, installation_id, archea_webhook_id)
                elif event_type in ["pull_request", "issues", "create", "delete"]:
                    await _handle_other_events(event_type, payload, db, user_id, archea_webhook_id)
                else:
                    await _handle_unknown_event(event_type, payload, db, user_id, archea_webhook_id)
                
                # Mark webhook as processed
                await database_service.update_webhook_status(
                    db, archea_webhook_id, WebhookStatus.PROCESSED
                )
                logger.info(f"Webhook {archea_webhook_id} processed successfully")
                
            except Exception as handler_error:
                logger.error(f"Error in webhook handler {archea_webhook_id}: {str(handler_error)}")
                await database_service.update_webhook_status(
                    db, archea_webhook_id, WebhookStatus.FAILED, str(handler_error)
                )
                # Don't re-raise - we've logged and updated status
                
    except Exception as e:
        logger.error(f"Background processing failed for webhook {archea_webhook_id}: {str(e)}")
        try:
            async for db in db_session_context():
                await database_service.update_webhook_status(
                    db, archea_webhook_id, WebhookStatus.FAILED, str(e)
                )
        except Exception as update_error:
            logger.error(f"Failed to update failed webhook status: {str(update_error)}")

async def _handle_installation_event(payload: dict, db: AsyncSession, user_id: str, archea_webhook_id: int):
    """Handle installation events with proper user account linking."""
    action = payload.get("action")
    installation = payload.get("installation", {})
    installation_id = installation.get("id")
    account = installation.get("account", {})
    username = account.get("login")
    account_type = account.get("type", "User")
    github_id = account.get("id")
    
    if not installation_id:
        logger.warning(f"No installation ID in installation event for user {user_id}")
        return
    
    try:
        if action in ["created", "added"]:
            # Get user
            user = await database_service.get_user_by_id(db, user_id)
            if not user:
                logger.error(f"User {user_id} not found for installation event")
                return
            
            # Create or update GitHub account
            github_account_data = {
                "user_id": user.id,
                "github_username": username,
                "github_id": github_id,
                "installation_id": installation_id,
                "account_type": account_type,
                "avatar_url": account.get("avatar_url"),
                "html_url": account.get("html_url"),
                "permissions": installation.get("permissions"),
                "is_active": True,
                "created_at": datetime.utcnow()
            }
            
            existing_account = await database_service.get_github_account_by_installation_id(db, installation_id)
            if existing_account:
                await database_service.update_github_account(db, installation_id, github_account_data)
                logger.info(f"Updated GitHub account for {username}")
            else:
                await database_service.create_github_account(db, github_account_data)
                logger.info(f"Created GitHub account for {username}")
            
            # Auto-sync repositories after installation
            try:
                await github_service.sync_all_repositories_and_branches(user_id, installation_id, db)
                logger.info(f"Auto-sync completed for {username}")
            except Exception as sync_error:
                logger.error(f"Auto-sync failed for {username}: {str(sync_error)}")
            
        elif action in ["deleted", "suspend"]:
            await database_service.update_github_account(
                db, installation_id, {
                    "is_active": False,
                    "suspended_at": datetime.utcnow()
                }
            )
            logger.info(f"Deactivated GitHub account for installation {installation_id}")
            
    except Exception as e:
        logger.error(f"Error handling installation event for user {user_id}: {str(e)}")
        raise

async def _handle_push_event(payload: dict, db: AsyncSession, user_id: str, installation_id: int, archea_webhook_id: int):
    """Handle push events with proper repository validation."""
    repository = payload.get("repository", {})
    repo_id = repository.get("id")
    repo_name = repository.get("name")
    repo_full_name = repository.get("full_name")
    
    if not repo_id:
        logger.warning(f"No repository ID in push event for user {user_id}")
        return
    
    try:
        # Get GitHub account
        github_account = await database_service.get_github_account_by_installation_id(db, installation_id)
        if not github_account:
            logger.warning(f"No GitHub account found for installation {installation_id}")
            return
        
        # Find or create repository
        repo = await database_service.get_repository_by_id_and_user_id(db, repo_id, github_account.user_id,installation_id)
        if not repo:
            repo_data = {
                "github_account_id": github_account.id,
                "repo_id": repo_id,
                "name": repo_name,
                "full_name": repo_full_name,
                "private": repository.get("private", False),
                "html_url": repository.get("html_url"),
                "description": repository.get("description"),
                "language": repository.get("language"),
                "default_branch": repository.get("default_branch", "main"),
            }
            repo = await database_service.create_repository(db, repo_data)
            logger.info(f"Created repository record for {repo_full_name}")
        
        # Handle push event
        await github_service.handle_push_event(payload, db, user_id,archea_webhook_id,installation_id,repo.id)
        logger.info(f"Processed push event for {repo_full_name}")
        
    except Exception as e:
        logger.error(f"Error handling push event for repo {repo_full_name}: {str(e)}")
        raise

async def _handle_other_events(event_type: str, payload: dict, db: AsyncSession, user_id: str, archea_webhook_id: int):
    """Handle other GitHub events."""
    logger.info(f"Processing {event_type} event for user {user_id}")
    # Implement specific logic for other event types as needed
    # For now, just log and mark as processed

async def _handle_unknown_event(event_type: str, payload: dict, db: AsyncSession, user_id: str, archea_webhook_id: int):
    """Handle unknown event types."""
    logger.warning(f"Unknown event type received: {event_type} for user {user_id}")
    await database_service.update_webhook_status(
        db, archea_webhook_id, WebhookStatus.UNKNOWN_EVENT
    )

@github_router.get(
    "/callback",
    summary="GitHub App Callback",
    response_model=GitHubCallbackResponse,
    responses={
        400: {"model": ErrorResponse},
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def github_app_callback(
    code: str = Query(..., description="OAuth code from GitHub"),
    installation_id: Optional[int] = Query(None, description="GitHub App installation ID"),  # Changed to Optional
    setup_action: Optional[str] = Query(None, description="Setup action from GitHub"),
    state: str = Query(..., description="Internal user ID passed during installation"),
    db: AsyncSession = Depends(get_async_db)
):
    """Handle GitHub OAuth callback for both App installation and regular OAuth flow."""
    """Handle GitHub App callback after installation with proper user linking."""
    try:
        
        if installation_id:
            # Get installation info from GitHub
            installation_info = await github_service.get_installation_info(installation_id)
            
            account = installation_info.get("account", {})
            username = account.get("login")
            account_type = account.get("type")
            github_id = account.get("id")

            if not username:
                raise HTTPException(
                    status_code=status.HTTP_400_BAD_REQUEST,
                    detail="Could not extract username from installation info"
                )
        else:
            # Exchange code for access token
                token_data = await github_service.exchange_code_for_token(code)
                if not token_data or 'access_token' not in token_data:
                    raise HTTPException(
                        status_code=status.HTTP_400_BAD_REQUEST,
                        detail="Failed to exchange code for access token"
                    )

                access_token = token_data['access_token']
                token_type = token_data.get('token_type', 'bearer')
                scope = token_data.get('scope', '')
                expires_in = token_data.get('expires_in')
                logger.info(access_token,"  access_token")
                # Get user info from GitHub using access token
                user_info = await github_service.get_user_info(access_token)
                if not user_info:
                    raise HTTPException(
                        status_code=status.HTTP_400_BAD_REQUEST,
                        detail="Failed to get user info from GitHub"
                    )

                username = user_info.get('login')
                github_id = user_info.get('id')
                account_type = user_info.get('type', 'User')

                if not username:
                    raise HTTPException(
                        status_code=status.HTTP_400_BAD_REQUEST,
                        detail="Could not extract username from GitHub user info"
                    )
                return GitHubCallbackResponse(
                    message="GitHub App installation hold completed successfully",
                    installation_id=1000,
                    setup_action=setup_action,
                    username=username,
                    account_type=account_type,
                    user_id=state
                )
        # Find user
        user = await database_service.get_user_by_id(db, int(state))
        if not user:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )
        
        # Create or update GitHub account
        github_account_data = {
            "user_id": user.id,
            "github_username": username,
            "github_id": github_id,
            "installation_id": installation_id,
            "github_login_code": code,
            "account_type": account_type,
            "avatar_url": account.get("avatar_url"),
            "html_url": account.get("html_url"),
            "permissions": installation_info.get("permissions"),
            "is_active": True,
            "created_at": datetime.utcnow()
        }
        
        existing_account = await database_service.get_github_account_by_installation_id(db, installation_id)
        if existing_account:
            await database_service.update_github_account(db, installation_id, github_account_data)
            logger.info(f"Updated GitHub account for {username}")
        else:
            await database_service.create_github_account(db, github_account_data)
            logger.info(f"Created GitHub account for {username}")

        return GitHubCallbackResponse(
            message="GitHub App installation completed successfully",
            installation_id=installation_id,
            setup_action=setup_action,
            username=username,
            account_type=account_type,
            user_id=state
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in GitHub callback: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error during GitHub App callback"
        )

@github_router.get(
    "/{user_id}/repositories",
    response_model=RepositoryListConfigResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_user_repositories(
    user_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """Fetch all repositories for a user."""
    try:
        # Get user's GitHub accounts
        github_accounts = await database_service.get_github_accounts_by_user_id(db, user_id)
        if not github_accounts:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="No GitHub accounts found for user"
            )
        
        # Get all repositories for user's GitHub accounts
        all_repositories = []
        for account in github_accounts:
            if account.is_active:
                repos = await database_service.get_repositories_by_account(db, account.id)
                all_repositories.extend(repos)
        
        # Format response
        formatted_repos = []
        for repo in all_repositories:
            formatted_repos.append(RepositoryInfo(
                id=repo.repo_id,
                name=repo.name,
                full_name=repo.full_name,
                private=repo.private,
                html_url=repo.html_url,
                description=repo.description,
                language=repo.language,
                updated_at=repo.updated_at or repo.created_at,
                visibility="private" if repo.private else "public",
                default_branch=repo.default_branch,
                archived=getattr(repo, 'archived', False)
            ))
        
        private_count = len([r for r in all_repositories if r.private])
        public_count = len(all_repositories) - private_count
        
        return RepositoryListConfigResponse(
            user_id=user_id,
            total_repositories=len(all_repositories),
            private_repositories=private_count,
            public_repositories=public_count,
            repositories=formatted_repos
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error fetching repositories for user {user_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error fetching repositories"
        )

@github_router.get(
    "/repositories/{repository_id}/branches",
    response_model=BranchListConfResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def get_repository_branches(
    repository_id: int,
    db: AsyncSession = Depends(get_async_db)
):
    """Get all branches for a specific repository."""
    try:
        # Get repository
        repository = await database_service.get_repository_by_archea_repo_id(db, repository_id)
        if not repository:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="Repository not found"
            )
        
        # Get branches
        branches = await database_service.get_branches_by_repository(db, repository.id)
        
        # Format response
        formatted_branches = []
        for branch in branches:
            formatted_branches.append(BranchInfo(
                id=branch.id,
                name=branch.name,
                commit_sha=branch.commit_sha,
                commit_message=branch.commit_message,
                commit_author=branch.commit_author,
                is_protected=branch.is_protected,
                last_synced_at=branch.last_synced_at
            ))
        
        return BranchListConfResponse(
            repository_id=repository.id,
            repository_name=repository.name,
            total_branches=len(branches),
            branches=formatted_branches
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting branches for repository {repository_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error fetching branches"
        )

@github_router.post(
    "/sync",
    response_model=ManualSyncResponse,
    responses={
        404: {"model": ErrorResponse},
        500: {"model": ErrorResponse}
    }
)
async def manual_sync_repositories(
    request: ManualSyncRequest,
    background_tasks: BackgroundTasks,
    db: AsyncSession = Depends(get_async_db)
):
    """Manually trigger repository sync for a user."""
    try:
        user_id = request.user_id
        
        # Get user's active GitHub accounts
        github_accounts = await database_service.get_github_accounts_by_user_id(db, user_id)
        active_accounts = [acc for acc in github_accounts if acc.is_active]
        
        if not active_accounts:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="No active GitHub installations found for user"
            )
        
        # Use the first active account
        account = active_accounts[0]
        
        # Create sync job
        job_data = {
            "user_id": user_id,
            "job_type": "manual_sync",
            "status": SyncJobStatus.RUNNING,
            "trigger_event": "manual",
            "started_at": datetime.utcnow()
        }
        sync_job = await database_service.create_sync_job(db, job_data)
        
        # Trigger sync in background
        background_tasks.add_task(
            github_service.sync_all_repositories_and_branches,
            user_id,
            account.installation_id,
            db,
            sync_job.id
        )
        
        return ManualSyncResponse(
            message=f"Sync initiated for user {user_id}",
            user_id=user_id,
            installation_id=account.installation_id,
            sync_job_id=sync_job.id,
            timestamp=datetime.utcnow()
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in manual sync for user {request.user_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error initiating sync"
        )

   
@github_router.post(
    "/webhook/test",
    response_model=WebhookTestResponse,
    responses={500: {"model": ErrorResponse}}
)
async def test_webhook(
    request: TestWebhookRequest,
    db: AsyncSession = Depends(get_async_db)
):
    """
    Test webhook endpoint - accepts event and payload in request body.
    Performs actual GitHub operations but stores in database.
    """
    try:
        event = request.event
        payload = request.payload
        installation_id = payload.get("installation", {}).get("id")
        # user_id = request.user_id
        user=await database_service.get_user_by_installation_id(db,installation_id)
        # Verify user exists
        # user = await database_service.get_user_by_id(db, user)
        if not user:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="User not found"
            )
        
        logger.info(f"TEST WEBHOOK - Processing event: {event} for user: {user.id}")
        
        actions_performed = []
        webhook_id = None

        # Store test webhook in database
        webhook_data = {
            "webhook_id": 0,  # Test webhook
            "event_type": event,
            "payload": payload,
            "headers": {"test": "true"},
            "delivery_id": f"test-{uuid.uuid4()}",
            "signature": "test-signature",
            "user_id": user.id,
            "installation_id":installation_id,
            "status": WebhookStatus.PROCESSED,
            "created_at": datetime.utcnow()
        }
        
        webhook = await database_service.create_webhook(db, webhook_data)

        archea_webhook_id = webhook.id
        
        actions_performed.append(f"✓ Created test webhook record (ID: {archea_webhook_id})")

        # Handle events (same logic as main webhook but with test flag)
        if event == "installation":
            await _handle_installation_event(payload, db, user.id, archea_webhook_id)
            actions_performed.append("✓ Processed installation event")
            
        elif event == "push":
            installation_id = payload.get("installation", {}).get("id")
            if installation_id:
                
                await _handle_push_event(payload, db, user.id, installation_id, archea_webhook_id)
                repository = payload.get("repository", {})
                repo_full_name = repository.get("full_name")
                ref = payload.get("ref", "")
                branch_name = ref.replace("refs/heads/", "") if ref.startswith("refs/heads/") else ref
                actions_performed.append(f"✓ Processed push event for {repo_full_name} (branch: {branch_name})")
            else:
                actions_performed.append("⚠️ No installation ID in push event payload")

        return WebhookTestResponse(
            status="test_executed",
            event=event,
            user_id=user.id,
            message="Webhook executed in test mode (data stored in PostgreSQL)",
            actions_performed=actions_performed,
            webhook_id=webhook_id,
            notes=[
                "✓ Webhook recorded in database",
                "✓ Actual GitHub operations performed",
                "✓ Data stored in PostgreSQL",
                "✓ Files stored in storage system"
            ]
        )
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in test webhook: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error executing test webhook"
        )
