import logging
import os
from logging.handlers import TimedRotatingFileHandler
from datetime import datetime, timezone
from app.core.config import settings


class UTCFormatter(logging.Formatter):
    """Formatter that ensures timestamps are in UTC."""
    converter = datetime.fromtimestamp

    def formatTime(self, record, datefmt=None):
        dt = datetime.fromtimestamp(record.created, tz=timezone.utc)
        if datefmt:
            return dt.strftime(datefmt)
        return dt.isoformat()


def setup_logger(name: str = "app") -> logging.Logger:
    """
    Setup and return a logger that:
      - Logs in UTC
      - Rotates daily
      - Keeps up to settings.LOG_BACKUP_DAYS
    """

    logger = logging.getLogger(name)
    if logger.handlers:
        return logger  # already configured

    # Create log directory if not exists
    os.makedirs(settings.APP_LOG_DIR, exist_ok=True)

    # Log file path
    log_file = os.path.join(settings.APP_LOG_DIR, settings.APP_LOG_FILE_NAME)

    # File handler (daily rotation, UTC, keep N days)
    file_handler = TimedRotatingFileHandler(
        log_file,
        when="midnight",
        interval=1,
        backupCount=settings.APP_LOG_BACKUP_DAYS,
        utc=True,
    )

    formatter = UTCFormatter(
        fmt="%(asctime)s | %(levelname)s | %(name)s | %(message)s",
        datefmt="%Y-%m-%dT%H:%M:%SZ",
    )

    file_handler.setFormatter(formatter)

    # Console handler
    console_handler = logging.StreamHandler()
    console_handler.setFormatter(formatter)

    # Attach handlers
    logger.setLevel(settings.APP_LOG_LEVEL.upper())
    logger.addHandler(file_handler)
    logger.addHandler(console_handler)
    logger.propagate = False

    return logger


# Global logger accessible anywhere
logger = setup_logger()
