from fastapi import Request
from starlette.middleware.base import BaseHTTPMiddleware
from time import time
from app.core.logger import logger


class LoggingMiddleware(BaseHTTPMiddleware):
    """
    Middleware to log each API request and response.
    Logs method, URL, status code, duration, and client IP.
    """

    async def dispatch(self, request: Request, call_next):
        start_time = time()
        client_host = request.client.host if request.client else "unknown"

        try:
            response = await call_next(request)
            status_code = response.status_code
        except Exception as e:
            status_code = 500
            logger.exception(f"Exception handling request: {e}")
            raise

        process_time = (time() - start_time) * 1000  # ms
        log_data = {
            "method": request.method,
            "url": request.url.path,
            "status_code": status_code,
            "duration_ms": f"{process_time:.2f}",
            "client_ip": client_host,
        }

        logger.info(
            "%(method)s %(url)s | Status: %(status_code)s | "
            "Duration: %(duration_ms)s ms | Client: %(client_ip)s",
            log_data,
        )

        return response
