# app/schemas/common_schemas.py
from pydantic import BaseModel, EmailStr, Field
from typing import List, Optional, Dict, Any
from datetime import datetime
from enum import Enum

# ===== ENUMS =====
class HealthStatus(str, Enum):
    HEALTHY = "healthy"
    DEGRADED = "degraded"
    UNHEALTHY = "unhealthy"

class SyncJobStatus(str, Enum):
    PENDING = "pending"
    RUNNING = "running"
    COMPLETED = "completed"
    FAILED = "failed"

class SyncJobType(str, Enum):
    FULL_SYNC = "full_sync"
    PUSH_SYNC = "push_sync"
    MANUAL_SYNC = "manual_sync"

class AccountType(str, Enum):
    USER = "User"
    ORGANIZATION = "Organization"

# ===== PAGINATION SCHEMAS =====
class PaginationParams(BaseModel):
    skip: int = Field(0, ge=0, description="Number of records to skip")
    limit: int = Field(100, ge=1, le=1000, description="Number of records to return")

class ActivityPagination(PaginationParams):
    limit: int = Field(50, ge=1, le=200, description="Number of activities to return")

# ===== FILTER SCHEMAS =====
class BaseFilter(BaseModel):
    is_active: Optional[bool] = None

class UserFilter(BaseFilter):
    email: Optional[EmailStr] = None
    username: Optional[str] = None

class GitHubAccountFilter(BaseFilter):
    github_username: Optional[str] = None
    account_type: Optional[AccountType] = None

class RepositoryFilter(BaseFilter):
    name: Optional[str] = None
    private: Optional[bool] = None
    language: Optional[str] = None

class SyncJobFilter(BaseFilter):
    job_type: Optional[SyncJobType] = None
    status: Optional[SyncJobStatus] = None
    trigger_event: Optional[str] = None

# ===== RESPONSE WRAPPERS =====
class ListResponse(BaseModel):
    total_count: int
    skip: int
    limit: int

class SuccessResponse(BaseModel):
    message: str
    status: str = "success"

class HealthCheckResponse(BaseModel):
    status: str
    timestamp: datetime
    version: str
    dependencies: Dict[str, str]