# app/schemas/sync_schemas.py
from pydantic import BaseModel, Field
from typing import Optional, List, Dict, Any
from datetime import datetime
from app.schemas.common_schemas import SyncJobStatus, SyncJobType, ListResponse

# ===== BASE SCHEMAS =====
class SyncJobBase(BaseModel):
    job_type: SyncJobType
    status: SyncJobStatus = SyncJobStatus.PENDING
    trigger_event: Optional[str] = Field(None, max_length=100)
    changes_summary: Optional[Dict[str, Any]] = None
    storage_path: Optional[str] = None
    download_url: Optional[str] = None
    error_message: Optional[str] = None

# ===== CREATE/UPDATE SCHEMAS =====
class SyncJobCreate(SyncJobBase):
    user_id: int = Field(..., gt=0)
    branch_id: Optional[int] = Field(None, gt=0)

class SyncJobUpdate(BaseModel):
    status: Optional[SyncJobStatus] = None
    changes_summary: Optional[Dict[str, Any]] = None
    storage_path: Optional[str] = None
    download_url: Optional[str] = None
    error_message: Optional[str] = None
    started_at: Optional[datetime] = None
    completed_at: Optional[datetime] = None

# ===== RESPONSE SCHEMAS =====
class SyncJobResponse(BaseModel):
    id: int
    user_id: int
    branch_id: Optional[int]
    job_type: SyncJobType
    status: SyncJobStatus
    trigger_event: Optional[str]
    changes_summary: Optional[Dict[str, Any]]
    storage_path: Optional[str]
    download_url: Optional[str]
    error_message: Optional[str]
    started_at: Optional[datetime]
    completed_at: Optional[datetime]
    created_at: datetime

    class Config:
        from_attributes = True

class SyncJobSummary(BaseModel):
    id: int
    job_type: SyncJobType
    status: SyncJobStatus
    trigger_event: Optional[str]
    started_at: Optional[datetime]
    completed_at: Optional[datetime]
    created_at: datetime

class SyncStatistics(BaseModel):
    github_accounts: int
    repositories: int
    branches: int
    total_sync_jobs: int
    completed_jobs: int
    failed_jobs: int
    running_jobs: int

class UserSyncStatusResponse(BaseModel):
    user_id: int
    statistics: SyncStatistics
    recent_sync_jobs: List[SyncJobSummary]

# ===== CODE CHANGE SCHEMAS =====
class CodeChangeBase(BaseModel):
    repo: str = Field(..., max_length=500)
    branch: str = Field(..., max_length=255)
    commit_hash: str = Field(..., min_length=7, max_length=255)
    commit_author: Optional[str] = Field(None, max_length=255)
    commit_message: Optional[str] = None
    changes_data: Dict[str, Any]
    total_files_changed: int = Field(0, ge=0)
    files_added: int = Field(0, ge=0)
    files_modified: int = Field(0, ge=0)
    files_deleted: int = Field(0, ge=0)

class CodeChangeCreate(CodeChangeBase):
    sync_job_id: int = Field(..., gt=0)

class CodeChangeResponse(BaseModel):
    id: int
    sync_job_id: int
    repo: str
    branch: str
    commit_hash: str
    commit_author: Optional[str]
    commit_timestamp: Optional[datetime]
    commit_message: Optional[str]
    changes_data: Dict[str, Any]
    total_files_changed: int
    files_added: int
    files_modified: int
    files_deleted: int
    created_at: datetime

    class Config:
        from_attributes = True

class ActivitySummary(BaseModel):
    id: int
    repo: str
    branch: str
    commit_hash: str
    commit_author: Optional[str]
    commit_timestamp: Optional[datetime]
    total_files_changed: int
    files_added: int
    files_modified: int
    files_deleted: int
    created_at: datetime

class UserActivityResponse(BaseModel):
    user_id: int
    total_activities: int
    recent_activities: List[ActivitySummary]