# app/services/github_service.py
import httpx
import hmac
import hashlib
import jwt
import time
import io
import zipfile
import base64
from typing import Optional, List, Dict, Any
from datetime import datetime
import pytz
import json
import re
from sqlalchemy import select
from app.core.config import settings
from app.core.logger import logger
from app.services.storage_service import storage_service
from app.services.postgress_db_service import pg_db_service as database_service
from app.db.postgress_db import get_async_db
from sqlalchemy.ext.asyncio import AsyncSession
from app.models.sync_models import SyncJob



class GitHubAppService:
    def __init__(self):
        self.app_id = settings.GITHUB_APP_ID
        self.private_key = settings.GITHUB_PRIVATE_KEY
        self.webhook_secret = settings.GITHUB_WEBHOOK_SECRET
        self.app_name = settings.APP_NAME

    # ===== AUTHENTICATION METHODS =====
    def generate_app_jwt(self) -> str:
        """Generate a JWT for GitHub App authentication."""
        payload = {
            "iat": int(time.time()),
            "exp": int(time.time()) + (10 * 60),
            "iss": self.app_id
        }
        
        private_key = self.private_key
        if not private_key.startswith('-----BEGIN RSA PRIVATE KEY-----'):
            private_key = private_key.replace('\\n', '\n')
        
        token = jwt.encode(payload, private_key, algorithm="RS256")
        logger.info("Generated new GitHub App JWT")
        return token
    
    async def exchange_code_for_token(self, code: str) -> Dict[str, Any]:
        """Exchange OAuth code for access token."""
        try:
            url = f"https://github.com/login/oauth/access_token"
            
            payload = {
                'client_id': settings.GITHUB_CLIENT_ID,
                'client_secret': settings.GITHUB_CLIENT_SECRET,
                'code': code
            }
            
            headers = {
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
            
            async with httpx.AsyncClient() as client:
                response = await client.post(url, json=payload, headers=headers)
                response.raise_for_status()
                token_data = response.json()
                
                if 'error' in token_data:
                    error_msg = token_data.get('error_description', token_data['error'])
                    logger.error(f"GitHub OAuth error: {error_msg}")
                    raise
                
                if 'access_token' not in token_data:
                    logger.error(f"Missing access_token in response: {token_data}")
                    raise
                
                logger.info(f"Successfully exchanged code for token")
                return token_data
                
        except httpx.HTTPStatusError as e:
            logger.error(f"GitHub OAuth token exchange failed: {e.response.status_code}")
            raise
        except Exception as e:
            logger.error(f"Failed to exchange code for token: {str(e)}")
            raise 

    async def get_user_info(self, access_token: str) -> Dict[str, Any]:
        """Get GitHub user information using access token."""
        try:
            url = f"https://api.github.com/user"
            
            headers = {
                "Authorization": f"Bearer {access_token}",
                "Accept": "application/vnd.github+json",
                "User-Agent": self.app_name
            }
            
            async with httpx.AsyncClient() as client:
                response = await client.get(url, headers=headers)
                response.raise_for_status()
                user_info = response.json()
                
                if 'login' not in user_info:
                    logger.error(f"Invalid user info response: {user_info}")
                    raise 
                
                logger.info(f"Retrieved user info for: {user_info.get('login')}")
                return user_info
                
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 401:
                logger.error(f"Invalid access token")
                raise 
            else:
                logger.error(f"GitHub user info failed: {e.response.status_code}")
                raise 
        except Exception as e:
            logger.error(f"Failed to fetch user info: {str(e)}")
            raise 

    async def get_installation_token(self, installation_id: int) -> str:
        """Exchange GitHub App JWT for an installation token."""
        jwt_token = self.generate_app_jwt()
        url = f"https://api.github.com/app/installations/{installation_id}/access_tokens"

        try:
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {jwt_token}",
                    "Accept": "application/vnd.github+json",
                    "User-Agent": self.app_name
                }
                response = await client.post(url, headers=headers)
                response.raise_for_status()
                data = response.json()
                
                installation_token = data["token"]
                logger.info(f"Generated new installation token for installation {installation_id}")
                return installation_token
                
        except httpx.HTTPStatusError as e:
            logger.error(f"HTTP error getting installation token: {e.response.status_code}")
            if e.response.status_code in [404, 403]:
                await self.clear_installation_cache_by_id(installation_id)
            raise
        except Exception as e:
            logger.error(f"Error getting installation token: {e}")
            raise

    # ===== INSTALLATION MANAGEMENT =====
    async def store_installation(self, installation_id: int, username: str, account_type: str = "User"):
        """Store installation information - now handled in webhook."""
        logger.info(f"Installation stored in PostgreSQL for {account_type}: {username}")

    
    async def find_installation_by_user(self, username: str) -> Optional[Dict[str, Any]]:
        """Find installation by scanning all installations for this user/org."""
        jwt_token = self.generate_app_jwt()
        url = "https://api.github.com/app/installations"
        
        try:
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {jwt_token}",
                    "Accept": "application/vnd.github+json",
                    "User-Agent": self.app_name
                }
                
                page = 1
                while True:
                    params = {"per_page": 100, "page": page}
                    response = await client.get(url, headers=headers, params=params)
                    response.raise_for_status()
                    
                    installations = response.json()
                    if not installations:
                        break
                    
                    for installation in installations:
                        account = installation.get("account", {})
                        if account.get("login") == username:
                            installation_id = installation.get("id")
                            account_type = account.get("type", "User")
                            await self.store_installation(installation_id, username, account_type)
                            logger.info(f"Found installation {installation_id} for {username} via API scan")
                            return installation
                    page += 1
                    
        except Exception as e:
            logger.error(f"Error finding installation for {username}: {e}")
            
        return None

    
    async def get_installation_info(self, installation_id: int) -> Dict[str, Any]:
        """Fetch installation information from GitHub API."""
        jwt_token = self.generate_app_jwt()
        url = f"https://api.github.com/app/installations/{installation_id}"
        
        async with httpx.AsyncClient() as client:
            headers = {
                "Authorization": f"Bearer {jwt_token}",
                "Accept": "application/vnd.github+json",
                "User-Agent": self.app_name
            }
            response = await client.get(url, headers=headers)
            response.raise_for_status()
            return response.json()

    

    # ===== REPOSITORY METHODS =====
    async def fetch_user_repositories(self, username: str) -> List[Dict[str, Any]]:
        """Fetch all repositories for a user/org using GitHub App installation token."""
        installation_id = await self.get_or_find_installation_id(username)
        if not installation_id:
            raise ValueError(f"No GitHub App installation found for {username}")

        token = await self.get_installation_token(installation_id)
        github_api_url = "https://api.github.com/installation/repositories"
        repos = []

        try:
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {token}",
                    "Accept": "application/vnd.github+json",
                    "User-Agent": self.app_name
                }

                page = 1
                while True:
                    params = {"per_page": 100, "page": page}
                    response = await client.get(github_api_url, headers=headers, params=params)
                    
                    if response.status_code == 401:
                        token = await self.get_installation_token(installation_id)
                        headers["Authorization"] = f"Bearer {token}"
                        response = await client.get(github_api_url, headers=headers, params=params)
                    
                    response.raise_for_status()
                    data = response.json()
                    
                    page_repos = data.get("repositories", [])
                    if not page_repos:
                        break
                        
                    repos.extend(page_repos)
                    page += 1

            return repos

        except httpx.HTTPStatusError as e:
            logger.error(f"GitHub API error for {username}: {e.response.status_code}")
            if e.response.status_code in [404, 403]:
                await self.clear_installation_cache(username)
            raise
        except Exception as e:
            logger.error(f"Error fetching repos for {username}: {e}")
            raise

    def format_repository_data(self, repos: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
        """Format repository data for API response."""
        return [
            {
                "id": repo["id"],
                "name": repo["name"],
                "full_name": repo["full_name"],
                "private": repo["private"],
                "html_url": repo["html_url"],
                "description": repo.get("description"),
                "language": repo.get("language"),
                "updated_at": repo["updated_at"],
                "visibility": "private" if repo["private"] else "public"
            }
            for repo in repos
        ]

    
    # ===== SYNC METHODS =====
    async def sync_all_repositories_and_branches(self, user_id: int, installation_id: int, db,job_id:int=None):
        """Automatically sync all repositories and all branches to storage."""
        try:
            logger.info(f"Starting automatic sync for {user_id} (installation: {installation_id})")
            
            token = await self.get_installation_token(installation_id)
            repos = await self._get_all_repositories(token)
            
            logger.info(f"Found {len(repos)} repositories for {user_id}")
            
            # Get GitHub account from database
            github_account = await database_service.get_github_account_by_installation_id(db, installation_id)
            if not github_account:
                logger.error(f"GitHub account not found for installation_id: {installation_id}")
                return

            # Create sync job
            user = await database_service.get_user_by_id(db, user_id)
            if user and not job_id:
                job_data = {
                    "user_id": user.id,
                    "job_type": "full_sync",
                    "status": "running",
                    "trigger_event": "auto_install"
                }
                sync_job = await database_service.create_sync_job(db, job_data)
            if job_id:
                sync_job= await database_service.get_sync_job_by_id(db,job_id)

            for repo in repos:
                try:
                    await self._sync_repo_branches(user_id, repo, token, db, github_account.id)
                except Exception as e:
                    logger.error(f"Failed to sync repo {repo['full_name']}: {e}")
                    continue

            # Update sync job status
            if user and 'sync_job' in locals():
                await database_service.update_sync_job_status(db, sync_job.id, "completed")
                    
            logger.info(f"Completed automatic sync for user_id :{user_id}")
            
        except Exception as e:
            # Update sync job status to failed
            if 'user' in locals() and 'sync_job' in locals():
                await database_service.update_sync_job_status(db, sync_job.id, "failed", str(e))
            logger.error(f"Error in automatic sync for user_id:{user_id}: {e}")
            raise

    async def _get_all_repositories(self, token: str) -> List[Dict[str, Any]]:
        """Get all repositories for this installation."""
        repos = []
        page = 1
        
        async with httpx.AsyncClient() as client:
            headers = {
                "Authorization": f"Bearer {token}",
                "Accept": "application/vnd.github+json",
                "User-Agent": self.app_name
            }

            while True:
                params = {"per_page": 100, "page": page}
                response = await client.get("https://api.github.com/installation/repositories", headers=headers, params=params)
                response.raise_for_status()
                data = response.json()
                
                page_repos = data.get("repositories", [])
                if not page_repos:
                    break
                    
                repos.extend(page_repos)
                page += 1

        return repos

    async def _sync_repo_branches(self, user_id: int, repo: dict, token: str, db, github_account_id: int):
        """Sync all branches of a repository."""
        repo_id = repo["id"]
        repo_name = repo["name"]
        repo_full_name = repo["full_name"]
        
        logger.info(f"Processing repository: {repo_full_name}")
        
        # Create or update repository in database
        repo_data = {
            "github_account_id": github_account_id,
            "repo_id": repo_id,
            "name": repo_name,
            "full_name": repo_full_name,
            "private": repo.get("private", False),
            "html_url": repo.get("html_url"),
            "description": repo.get("description"),
            "language": repo.get("language"),
            "default_branch": repo.get("default_branch", "main"),
            "last_synced_at": datetime.now()
        }
        
        existing_repo = await database_service.get_repository_by_id(db, repo_id,github_account_id)
        if existing_repo:
            # Update existing repository
            for key, value in repo_data.items():
                setattr(existing_repo, key, value)
            await db.commit()
            repository = existing_repo
        else:
            # Create new repository
            repository = await database_service.create_repository(db, repo_data)
        
        branches = await self._get_all_repo_branches(repo_id, token)
        logger.info(f"Found {len(branches)} branches for {repo_full_name}")
        
        for branch in branches:
            try:
                branch_name = branch["name"]
                await self._download_and_store_branch(user_id, repo, branch_name, token, db, repository.id)
            except Exception as e:
                logger.error(f"Failed to download branch {branch_name} for repo {repo_full_name}: {e}")
                continue
                
        logger.info(f"Completed syncing all branches for {repo_full_name}")

    async def _get_all_repo_branches(self, repo_id: int, token: str) -> List[Dict[str, Any]]:
        """Get all branches for a repository."""
        branches = []
        page = 1
        
        async with httpx.AsyncClient() as client:
            headers = {
                "Authorization": f"Bearer {token}",
                "Accept": "application/vnd.github+json",
                "User-Agent": self.app_name
            }
            
            while True:
                params = {"per_page": 100, "page": page}
                branches_url = f"https://api.github.com/repositories/{repo_id}/branches"
                
                response = await client.get(branches_url, headers=headers, params=params)
                response.raise_for_status()
                page_branches = response.json()
                
                if not page_branches:
                    break
                    
                branches.extend(page_branches)
                page += 1
    
        return branches

    async def _download_and_store_branch(self, user_id: int, repo: dict, branch_name: str, token: str, db, repository_id: int):
        """Download and store a specific branch."""
        repo_id = repo["id"]
        repo_name = repo["name"]
        repo_full_name = repo["full_name"]
        
        logger.info(f"Downloading branch: {repo_full_name} (branch: {branch_name})")
        
        download_url = f"https://api.github.com/repos/{repo_full_name}/zipball/{branch_name}"
        
        async with httpx.AsyncClient() as client:
            headers = {
                "Authorization": f"Bearer {token}",
                "Accept": "application/vnd.github+json",
                "User-Agent": self.app_name
            }
            
            download_response = await client.get(download_url, headers=headers, follow_redirects=True)
            
            if download_response.status_code == 404:
                logger.warning(f"Branch {branch_name} not found in {repo_full_name}, skipping")
                return
                
            download_response.raise_for_status()

            archive_data = io.BytesIO(download_response.content)
            filename = f"archaea_user_id-{user_id}_repo_id-{repo_id}_branch_name-{branch_name}_code.zip"
            
            storage_result = await storage_service.upload_repo_archive(
                repo_name=repo_name,
                archive_data=archive_data,
                content_type="application/zip",
                metadata={
                    "repo_id": str(repo_id),
                    "repo_full_name": repo_full_name,
                    "user_id": user_id,
                    "branch": branch_name,
                    "downloaded_at": datetime.now().isoformat(),
                    "sync_type": "auto_install"
                },
                custom_filename=filename
            )
            
            logger.info(f"Successfully synced {repo_full_name} (branch: {branch_name})")
            
            # Create or update branch in database
            branch_data = {
                "repository_id": repository_id,
                "name": branch_name,
                "commit_sha": "Need actual",
                "last_synced_at": datetime.now()
            }
            
            existing_branches = await database_service.get_branches_by_repository(db, repository_id)
            existing_branch = next((b for b in existing_branches if b.name == branch_name), None)
            
            if existing_branch:
                # Update existing branch
                for key, value in branch_data.items():
                    setattr(existing_branch, key, value)
                await db.commit()
            else:
                # Create new branch
                await database_service.create_branch(db, branch_data)

    # ===== PUSH EVENT HANDLING =====
    async def handle_push_event(self, payload: dict, db:AsyncSession,user_id:int,archea_webhook_id:int,installation_id:int,repository_id:int):
        """Handle push events to update full branch code and changed files."""
        try:
            repository = payload.get("repository", {})
            repo_id = repository.get("id")
            repo_full_name = repository.get("full_name")
            repo_name = repository.get("name")
            owner = repository.get("owner", {})
            username = owner.get("login")
            ref = payload.get("ref", "")
            
            branch_name = ref.replace("refs/heads/", "") if ref.startswith("refs/heads/") else ref
            
            # installation_id = await database_service.get_installation_id_by_repo_and_user(db,repo_id,user_id)
            # if not installation_id:
            #     logger.error(f"No installation found for user: {username}")
            #     return
            branch_id=await database_service.get_branch_id_installation_and_name(db,branch_name=branch_name,installation_id=installation_id,repository_id=repository_id)

            if branch_id:
                await database_service.update_webhook_branch(db,archea_webhook_id,branch_id)

            token = await self.get_installation_token(installation_id)
            
            logger.info(f"Processing push event for {repo_full_name} (branch: {branch_name})")
            
            # Download full branch code
            await self._download_full_branch_code(user_id, repo_id, repo_name, repo_full_name, branch_name, token)
            githib_account=await database_service.get_github_account_by_installation_id(db,installation_id)
            # Process changed files in the new format
            await self._process_changed_files_new_format(user_id, repo_id, repo_name, repo_full_name, branch_name, payload, token, db,githib_account.id)
            
            logger.info(f"Completed processing push event for {repo_full_name} (branch: {branch_name})")
            
        except Exception as e:
            logger.error(f"Error handling push event: {e}")
            raise

    async def _download_full_branch_code(self, user_id: int, repo_id: int, repo_name: str, repo_full_name: str, branch_name: str, token: str):
        """Download full branch code."""
        try:
            logger.info(f"Downloading full branch code: {repo_full_name} (branch: {branch_name})")
            
            download_url = f"https://api.github.com/repos/{repo_full_name}/zipball/{branch_name}"
            
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {token}",
                    "Accept": "application/vnd.github+json",
                    "User-Agent": self.app_name
                }
                
                download_response = await client.get(download_url, headers=headers, follow_redirects=True)
                
                if download_response.status_code == 404:
                    logger.warning(f"Branch {branch_name} not found in {repo_full_name}")
                    return
                    
                download_response.raise_for_status()

                archive_data = io.BytesIO(download_response.content)
                filename = f"archaea_user_id-{user_id}_repo_id-{repo_id}_branch_name-{branch_name}_code.zip"
                
                storage_result = await storage_service.upload_repo_archive(
                    repo_name=repo_name,
                    archive_data=archive_data,
                    content_type="application/zip",
                    metadata={
                        "repo_id": str(repo_id),
                        "repo_full_name": repo_full_name,
                        "user_id": user_id,
                        "branch": branch_name,
                        "downloaded_at": datetime.now().isoformat(),
                        "sync_type": "push_full_branch"
                    },
                    custom_filename=filename
                )
                
                logger.info(f"Successfully saved full branch code: {filename}")
                
        except Exception as e:
            logger.error(f"Error downloading full branch code for {repo_full_name}: {e}")
            raise

    async def _process_changed_files_new_format(self, user_id: int, repo_id: int, repo_name: str, repo_full_name: str, branch_name: str, payload: dict, token: str, db:AsyncSession,github_account_id:int):
        """Process changed files in the new unified format."""
        try:
            commits = payload.get("commits", [])
            if not commits:
                logger.info(f"No commits found in push event for {repo_full_name}")
                return

            before_sha = payload.get("before")
            after_sha = payload.get("after")
            
            if not before_sha or before_sha == "0000000000000000000000000000000000000000":
                logger.info(f"Initial commit or no previous SHA for {repo_full_name}, skipping diff")
                return

            logger.info(f"Getting diff between {before_sha} and {after_sha} for {repo_full_name}")

            # Get the commit details for the latest commit
            latest_commit = commits[-1]
            commit_data = await self._get_commit_details(repo_full_name, after_sha, token)
            # Get structured diff
            structured_diff = await self._get_structured_diff(repo_full_name, before_sha, after_sha, token)
            # print(structured_diff)
            # Create the new format response
            changes_data = self._create_new_format_response(
                repo_full_name=repo_full_name,
                branch_name=branch_name,
                commit_data=commit_data,
                structured_diff=structured_diff
            )
            # print(changes_data)
            
            # Store the changes data in PostgreSQL
            await self._store_changes_data_postgres(user_id, repo_id, repo_name, branch_name, after_sha, changes_data, db, commit_data,github_account_id)
            
        except Exception as e:
            logger.error(f"Error processing changed files for {repo_full_name}: {e}")
            raise

    async def _get_commit_details(self, repo_full_name: str, commit_sha: str, token: str) -> Dict[str, Any]:
        """Get detailed commit information."""
        try:
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {token}",
                    "Accept": "application/vnd.github+json",
                    "User-Agent": self.app_name
                }
                
                commit_url = f"https://api.github.com/repos/{repo_full_name}/commits/{commit_sha}"
                response = await client.get(commit_url, headers=headers)
                response.raise_for_status()
                
                commit_data = response.json()
                return {
                    "hash": commit_data["sha"],
                    "author": commit_data["commit"]["author"]["name"],
                    "timestamp": commit_data["commit"]["author"]["date"],
                    "message": commit_data["commit"]["message"]
                }
                
        except Exception as e:
            logger.error(f"Error getting commit details for {commit_sha}: {e}")
            raise

    async def _get_structured_diff(self, repo_full_name: str, before_sha: str, after_sha: str, token: str) -> List[Dict[str, Any]]:
        """Get structured diff between two commits."""
        try:
            async with httpx.AsyncClient() as client:
                headers = {
                    "Authorization": f"Bearer {token}",
                    "Accept": "application/vnd.github.v3.diff",
                    "X-GitHub-Api-Version": "2022-11-28",
                    "User-Agent": self.app_name
                }
                
                compare_url = f"https://api.github.com/repos/{repo_full_name}/compare/{before_sha}...{after_sha}"
                response = await client.get(compare_url, headers=headers)
                response.raise_for_status()
                
                diff_content = response.text
                return await self._parse_diff_to_new_format(diff_content)
                    
        except Exception as e:
            logger.error(f"Error getting structured diff for {repo_full_name}: {e}")
            raise

    async def _parse_diff_to_new_format(self, diff_content: str) -> List[Dict[str, Any]]:
        """Parse unified diff format into the new structured format."""
        changes = []
        lines = diff_content.split('\n')
        
        current_file = None
        current_change = None
        
        i = 0
        while i < len(lines):
            line = lines[i]
            
            if line.startswith('diff --git'):
                # Save previous file change if exists
                if current_change:
                    changes.append(current_change)
                
                # Extract file paths
                parts = line.split()
                if len(parts) >= 3:
                    old_file = parts[2].replace('a/', '')
                    new_file = parts[3].replace('b/', '')
                    current_file = new_file if new_file != '/dev/null' else old_file
                
                # Determine change type
                change_type = "modified"
                i += 1
                
                # Check for file mode changes or new/deleted files
                while i < len(lines) and not lines[i].startswith('@@'):
                    if lines[i].startswith('new file'):
                        change_type = "added"
                    elif lines[i].startswith('deleted file'):
                        change_type = "deleted"
                    i += 1
                
                # Create new change object
                current_change = {
                    "file_name": current_file.split('/')[-1] if current_file else "unknown",
                    "file_path": current_file,
                    "change_type": change_type,
                    "diffs": []
                }
                
            elif line.startswith('@@') and current_change:
                hunk_header = line[2:].strip()
                hunk_info = self._parse_hunk_header(hunk_header)
                
                current_hunk = {
                    "change_from_line_start": hunk_info['old_start'],
                    "change_to_line_end": hunk_info['old_end'],
                    "change_new_line_start": hunk_info['new_start'],
                    "change_new_line_end": hunk_info['new_end'],
                    "old_lines": [],
                    "new_lines": []
                }
                
                i += 1
                
                # Process hunk content
                while i < len(lines) and not lines[i].startswith('@@') and not lines[i].startswith('diff --git'):
                    content_line = lines[i]
                    
                    if content_line.startswith(' '):
                        # Context line - exists in both versions
                        current_hunk["old_lines"].append(content_line[1:])
                        current_hunk["new_lines"].append(content_line[1:])
                    elif content_line.startswith('-'):
                        # Removed line - only in old version
                        current_hunk["old_lines"].append(content_line[1:])
                    elif content_line.startswith('+'):
                        # Added line - only in new version
                        current_hunk["new_lines"].append(content_line[1:])
                    elif content_line.startswith('\\'):
                        # Handle \ No newline at end of file
                        pass
                    else:
                        break
                        
                    i += 1
                
                # Clean up empty lines and validate
                if current_hunk["old_lines"] or current_hunk["new_lines"]:
                    current_hunk["old_lines"] = [line for line in current_hunk["old_lines"] if line.strip()]
                    current_hunk["new_lines"] = [line for line in current_hunk["new_lines"] if line.strip()]
                    
                    # Adjust line numbers for added/deleted files
                    if current_change["change_type"] == "added":
                        current_hunk["change_from_line_start"] = 0
                        current_hunk["change_to_line_end"] = 0
                    elif current_change["change_type"] == "deleted":
                        current_hunk["change_new_line_start"] = 0
                        current_hunk["change_new_line_end"] = 0
                    
                    current_change["diffs"].append(current_hunk)
                
            else:
                i += 1
        
        # Don't forget the last change
        if current_change:
            changes.append(current_change)
        return changes

    def _create_new_format_response(self, repo_full_name: str, branch_name: str, commit_data: Dict[str, Any], structured_diff: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Create the new format response."""
        return {
            "repo": f"github.com/{repo_full_name}",
            "branch": branch_name,
            "commit": commit_data,
            "changes": structured_diff
        }

    async def _store_changes_data_postgres(self, user_id: str, repo_id: int, repo_name: str, branch_name: str, commit_sha: str, changes_data: Dict[str, Any], db, commit_data: Dict[str, Any],github_account_id:int):
        """Store the changes data in PostgreSQL."""
        try:
            # Find user
            user = await database_service.get_user_by_id(db, user_id)
            if not user:
                logger.error(f"User not found for username: {user_id}")
                return

            # Find repository and branch
            repository = await database_service.get_repository_by_id(db, repo_id,github_account_id)
            if not repository:
                logger.error(f"Repository not found for repo_id: {repo_id}")
                return

            branches = await database_service.get_branches_by_repository(db, repository.id)
            branch = next((b for b in branches if b.name == branch_name), None)
            
            if not branch:
                logger.error(f"Branch not found: {branch_name}")
                return
            # Create sync job
            job_data = {
                "user_id": user.id,
                "branch_id": branch.id,
                "job_type": "push_sync",
                "status": "completed",
                "trigger_event": "webhook",
                "changes_summary": {
                    "total_files_changed": len(changes_data["changes"]),
                    "files_added": len([c for c in changes_data["changes"] if c["change_type"] == "added"]),
                    "files_modified": len([c for c in changes_data["changes"] if c["change_type"] == "modified"]),
                    "files_deleted": len([c for c in changes_data["changes"] if c["change_type"] == "deleted"])
                },
                "storage_path": f"{user_id}_{repo_id}_{branch_name}_{commit_sha[:7]}_changes.json",
                "download_url": f"storage/{user_id}_{repo_id}_{branch_name}_{commit_sha[:7]}_changes.json"
            }
            sync_job = await database_service.create_sync_job(db, job_data)

            logger.info(f"JobId {sync_job.id} for user_id-{user_id} repo_id-{repo_id} github_account_id-{github_account_id}")

            # Create code change record
            change_data = {
                "sync_job_id": sync_job.id,
                "repo": changes_data["repo"],
                "branch": branch_name,
                "commit_hash": commit_sha,
                "commit_author": commit_data["author"],
                "commit_timestamp": datetime.fromisoformat(commit_data["timestamp"].replace('Z', '+00:00')),
                "commit_message": commit_data["message"],
                "changes_data": changes_data,
                "total_files_changed": len(changes_data["changes"]),
                "files_added": len([c for c in changes_data["changes"] if c["change_type"] == "added"]),
                "files_modified": len([c for c in changes_data["changes"] if c["change_type"] == "modified"]),
                "files_deleted": len([c for c in changes_data["changes"] if c["change_type"] == "deleted"])
            }
            response = await database_service.create_code_change(db, change_data)
            logger.info(f"Successfully stored changes data codechangeId: {response.id} in PostgreSQL for {user_id}/{repo_name}/{branch_name}")
            
        except Exception as e:
            logger.error(f"Error storing changes data in PostgreSQL: {e}")
            raise

    def _parse_hunk_header(self, hunk_header: str) -> Dict[str, int]:
        """Parse hunk header like @@ -1,5 +1,6 @@ into correct line numbers."""
        pattern = r'@@ -(\d+)(?:,(\d+))? \+(\d+)(?:,(\d+))? @@'
        match = re.match(pattern, hunk_header)
        
        if match:
            old_start = int(match.group(1))
            old_count = int(match.group(2)) if match.group(2) else 1
            new_start = int(match.group(3))
            new_count = int(match.group(4)) if match.group(4) else 1
            
            old_end = old_start + old_count - 1
            new_end = new_start + new_count - 1
            
            return {
                'old_start': old_start,
                'old_count': old_count,
                'old_end': old_end,
                'new_start': new_start,
                'new_count': new_count,
                'new_end': new_end
            }
        
        # Fallback to manual parsing
        return self._parse_hunk_header_manual(hunk_header)

    def _parse_hunk_header_manual(self, hunk_header: str) -> Dict[str, int]:
        """Manual parsing for hunk headers as fallback."""
        content = hunk_header.strip('@ ')
        parts = content.split()
        
        if len(parts) < 2:
            return self._get_default_hunk_info()
        
        old_range = parts[0]
        new_range = parts[1]
        
        old_parts = old_range.lstrip('-').split(',')
        old_start = int(old_parts[0])
        old_count = int(old_parts[1]) if len(old_parts) > 1 else 1
        
        new_parts = new_range.lstrip('+').split(',')
        new_start = int(new_parts[0])
        new_count = int(new_parts[1]) if len(new_parts) > 1 else 1
        
        old_end = old_start + old_count - 1
        new_end = new_start + new_count - 1
        
        return {
            'old_start': old_start,
            'old_count': old_count,
            'old_end': old_end,
            'new_start': new_start,
            'new_count': new_count,
            'new_end': new_end
        }

    def _get_default_hunk_info(self) -> Dict[str, int]:
        """Return default hunk info when parsing fails."""
        return {
            'old_start': 1,
            'old_count': 0,
            'old_end': 0,
            'new_start': 1,
            'new_count': 0,
            'new_end': 0
        }

    # ===== UTILITY METHODS =====
    async def get_new_installation_user(self, user_id: str) -> str:
        """Generate installation URL for user."""
        install_url = (
            "https://github.com/apps/archaea-repo-sync/installations/new"
            f"?state={user_id}"
        )
        return install_url

    def verify_webhook_signature(self, body_bytes: bytes, signature: str) -> bool:
        """Verify GitHub webhook signature."""
        if not signature:
            return False

        mac = hmac.new(
            key=self.webhook_secret.encode(),
            msg=body_bytes,
            digestmod=hashlib.sha256
        )
        expected_signature = f"sha256={mac.hexdigest()}"
        return hmac.compare_digest(expected_signature, signature)

    async def clear_installation_cache_by_id(self, installation_id: int):
        """Clear installation cache by installation ID."""
        # This is now handled in PostgreSQL
        logger.info(f"Cleared installation cache for ID {installation_id}")

    async def clear_installation_cache(self, username: str):
        """Clear installation cache for a username."""
        # This is now handled in PostgreSQL
        logger.info(f"Cleared installation cache for user: {username}")

    async def get_sync_status(self, user_id: str,limit:int=10) -> Dict[str, Any]:
        """Get sync status for a user."""
        
        async for db in get_async_db():
            user = await database_service.get_user_by_id(db, user_id)
            if not user:
                return {
                    "user_id": user_id,
                    "total_synced_branches": 0,
                    "synced_repositories": []
                }

            # Get recent sync jobs
            result = await db.execute(
                select(SyncJob)
                .where(SyncJob.user_id == user.id)
                .order_by(SyncJob.created_at.desc())
                .limit(limit)
            )
            sync_jobs = result.scalars().all()
            
            
            return sync_jobs

    async def get_push_history(self, username: str, repo_id: Optional[int] = None, branch: Optional[str] = None) -> Dict[str, Any]:
        """Get push history for a user."""
        
        async for db in get_async_db():
            user = await database_service.get_user_by_username(db, username)
            if not user:
                return {
                    "username": username,
                    "total_push_events": 0,
                    "push_history": []
                }

            # Get code changes
            code_changes = await database_service.get_code_changes_by_user(db, user.id, limit=50)
            
            push_history = []
            for change in code_changes:
                push_history.append({
                    "repo_id": change.sync_job.branch.repository.repo_id if change.sync_job and change.sync_job.branch else None,
                    "repo_name": change.repo.split("/")[-1] if change.repo else None,
                    "branch": change.branch,
                    "commit_hash": change.commit_hash,
                    "commit_message": change.commit_message,
                    "total_files_changed": change.total_files_changed,
                    "synced_at": change.created_at.isoformat() if change.created_at else None
                })
            
            return {
                "username": username,
                "total_push_events": len(push_history),
                "push_history": push_history
            }


# Create service instance
github_service = GitHubAppService()