import redis
from app.core.config import settings
from app.core.logger import logger

class RedisClient:
    def __init__(self):
        try:
            self.r = redis.Redis(
                host=settings.REDIS_HOST,
                port=settings.REDIS_PORT,
                db=settings.REDIS_DB,
                password=settings.REDIS_PASSWORD,
                decode_responses=True
            )
            # Test the connection
            if self.r.ping():
                logger.info(f"Connected to Redis at {settings.REDIS_HOST}:{settings.REDIS_PORT} successfully")
            else:
                logger.warning(f"Redis connection to {settings.REDIS_HOST}:{settings.REDIS_PORT} failed")
        except redis.RedisError as e:
            logger.error(f"Error connecting to Redis: {e}")
            raise

    # -----------------------------
    # GitHub username/token methods
    # -----------------------------
    def store_user_token(self, username: str, token: str):
        """Store GitHub username and access token"""
        self.r.set(f"github:{username}:token", token)

    def get_user_token(self, username: str):
        """Retrieve GitHub access token by username"""
        return self.r.get(f"github:{username}:token")

    def delete_user_token(self, username: str):
        """Delete a GitHub token"""
        self.r.delete(f"github:{username}:token")

    # -----------------------------
    # Generic Redis methods
    # -----------------------------
    def set_key(self, key: str, value: str, expire: int = None):
        """Set any key/value in Redis"""
        self.r.set(name=key, value=value, ex=expire)

    def get_key(self, key: str):
        """Get any key from Redis"""
        return self.r.get(key)

    def delete_key(self, key: str):
        """Delete a key from Redis"""
        result = self.r.delete(key)
        if result:
            logger.info(f"Deleted key: {key}")
        else:
            logger.warning(f"Key not found for deletion: {key}")
        return result

    def keys(self, pattern: str):
        """Find keys matching a pattern"""
        return self.r.keys(pattern)

    def exists(self, key: str) -> bool:
        """Check if a key exists"""
        return self.r.exists(key) > 0

    def expire(self, key: str, seconds: int):
        """Set expiration time for a key"""
        return self.r.expire(key, seconds)

    def ttl(self, key: str) -> int:
        """Get time to live for a key"""
        return self.r.ttl(key)
    def is_connected(self)->bool:
        try:
            if self.r.ping():
                return True
        except:
            pass
        return False
                


# -----------------------------
# Usage example
# -----------------------------
redis_client = RedisClient()
# redis_client.store_user_token("octocat", "gho_123456...")
# token = redis_client.get_user_token("octocat")
# redis_client.delete_key("github:installation:octocat")