import os
import uuid
import shutil
import tempfile
from typing import Optional, Dict, Any, BinaryIO
from datetime import datetime, timedelta
from fastapi import HTTPException
import boto3
from botocore.exceptions import ClientError, NoCredentialsError

from app.core.config import settings
from app.core.logger import logger

class StorageService:
    def __init__(self):
        self.storage_type = settings.STORAGE_TYPE  # 'local' or 's3'
        self.local_storage_path = settings.LOCAL_STORAGE_PATH
        
        # S3 configuration
        self.s3_bucket = settings.S3_BUCKET_NAME
        self.s3_region = settings.S3_REGION
        self.s3_presigned_url_expiry = getattr(settings, 'S3_PRESIGNED_URL_EXPIRY', 3600)  # 1 hour default
        
        # Initialize S3 client if using S3
        if self.storage_type == 's3':
            try:
                self.s3_client = boto3.client(
                    's3',
                    aws_access_key_id=settings.AWS_ACCESS_KEY_ID,
                    aws_secret_access_key=settings.AWS_SECRET_ACCESS_KEY,
                    region_name=self.s3_region
                )
                logger.info(f"Initialized S3 client for bucket: {self.s3_bucket}")
            except NoCredentialsError:
                logger.error("AWS credentials not found")
                raise
        else:
            # Ensure local storage directory exists
            os.makedirs(self.local_storage_path, exist_ok=True)
            logger.info(f"Initialized local storage at: {self.local_storage_path}")

    async def upload_repo_archive(
        self, 
        repo_name: str, 
        archive_data: BinaryIO, 
        content_type: str = "application/zip",
        metadata: Optional[Dict[str, str]] = None,
        custom_filename: Optional[str] = None  # Add this parameter
    ) -> Dict[str, Any]:
        """
        Upload repository archive to storage and return download URL.
        
        Args:
            repo_name: Name of the repository
            archive_data: File-like object containing the archive data
            content_type: MIME type of the archive
            metadata: Additional metadata for the file
            custom_filename: Custom filename/path for storage (optional)
            
        Returns:
            Dictionary containing storage info and download URL
        """
        # Generate unique filename or use custom filename
        if custom_filename:
            filename = custom_filename
        else:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            filename = f"{repo_name}_{timestamp}_{uuid.uuid4().hex[:8]}.zip"
        
        if self.storage_type == 's3':
            return await self._upload_to_s3(filename, archive_data, content_type, metadata)
        else:
            return await self._upload_to_local(filename, archive_data, metadata)

    async def _upload_to_s3(
        self, 
        filename: str, 
        archive_data: BinaryIO, 
        content_type: str,
        metadata: Optional[Dict[str, str]] = None
    ) -> Dict[str, Any]:
        """
        Upload file to S3 and generate presigned URL.
        """
        try:
            # Upload to S3
            extra_args = {
                'ContentType': content_type,
            }
            
            if metadata:
                extra_args['Metadata'] = metadata
            
            # Reset the file pointer to beginning
            archive_data.seek(0)
            
            self.s3_client.upload_fileobj(
                archive_data,
                self.s3_bucket,
                filename,
                ExtraArgs=extra_args
            )
            
            logger.info(f"Successfully uploaded {filename} to S3 bucket {self.s3_bucket}")
            
            # Generate presigned URL for download
            download_url = self._generate_s3_presigned_url(filename)
            
            return {
                "storage_type": "s3",
                "filename": filename,
                "bucket": self.s3_bucket,
                "download_url": download_url,
                "expires_in": self.s3_presigned_url_expiry,
                "uploaded_at": datetime.now().isoformat(),
                "s3_path": f"s3://{self.s3_bucket}/{filename}"
            }
            
        except ClientError as e:
            logger.error(f"S3 upload error: {e}")
            raise HTTPException(status_code=500, detail=f"S3 upload failed: {str(e)}")
        except Exception as e:
            logger.error(f"Unexpected error during S3 upload: {e}")
            raise HTTPException(status_code=500, detail=f"Upload failed: {str(e)}")

    async def _upload_to_local(
        self, 
        filename: str, 
        archive_data: BinaryIO,
        metadata: Optional[Dict[str, str]] = None
    ) -> Dict[str, Any]:
        """
        Save file to local filesystem and generate local URL.
        """
        try:
            # For local storage, we need to handle the directory structure
            file_path = os.path.join(self.local_storage_path, filename)
            
            # Create directories if they don't exist
            os.makedirs(os.path.dirname(file_path), exist_ok=True)
            
            # Reset the file pointer to beginning
            archive_data.seek(0)
            
            # Save the file
            with open(file_path, 'wb') as f:
                shutil.copyfileobj(archive_data, f)
            
            logger.info(f"Successfully saved {filename} to local storage: {file_path}")
            
            # Generate local URL (relative path for the API to serve)
            download_url = f"/storage/download/{filename}"
            
            return {
                "storage_type": "local",
                "filename": filename,
                "file_path": file_path,
                "download_url": download_url,
                "uploaded_at": datetime.now().isoformat(),
                "local_path": file_path
            }
            
        except Exception as e:
            logger.error(f"Local storage upload error: {e}")
            raise HTTPException(status_code=500, detail=f"Local storage upload failed: {str(e)}")
    async def _upload_to_local(
        self, 
        filename: str, 
        archive_data: BinaryIO,
        metadata: Optional[Dict[str, str]] = None
    ) -> Dict[str, Any]:
        """
        Save file to local filesystem and generate local URL.
        """
        try:
            file_path = os.path.join(self.local_storage_path, filename)
            
            # Save the file
            with open(file_path, 'wb') as f:
                shutil.copyfileobj(archive_data, f)
            
            logger.info(f"Successfully saved {filename} to local storage: {file_path}")
            
            # Generate local URL (relative path for the API to serve)
            download_url = f"/storage/download/{filename}"
            
            return {
                "storage_type": "local",
                "filename": filename,
                "file_path": file_path,
                "download_url": download_url,
                "uploaded_at": datetime.now().isoformat()
            }
            
        except Exception as e:
            logger.error(f"Local storage upload error: {e}")
            raise HTTPException(status_code=500, detail=f"Local storage upload failed: {str(e)}")

    def _generate_s3_presigned_url(self, filename: str) -> str:
        """
        Generate presigned URL for S3 object download.
        """
        try:
            url = self.s3_client.generate_presigned_url(
                'get_object',
                Params={
                    'Bucket': self.s3_bucket,
                    'Key': filename
                },
                ExpiresIn=self.s3_presigned_url_expiry
            )
            return url
        except ClientError as e:
            logger.error(f"Error generating presigned URL: {e}")
            raise HTTPException(status_code=500, detail="Failed to generate download URL")

    async def download_file(self, filename: str) -> Dict[str, Any]:
        """
        Get download information for a file.
        """
        if self.storage_type == 's3':
            return await self._get_s3_download_info(filename)
        else:
            return await self._get_local_download_info(filename)

    async def _get_s3_download_info(self, filename: str) -> Dict[str, Any]:
        """
        Get S3 file download information with fresh presigned URL.
        """
        try:
            # Check if file exists
            self.s3_client.head_object(Bucket=self.s3_bucket, Key=filename)
            
            # Generate fresh presigned URL
            download_url = self._generate_s3_presigned_url(filename)
            
            return {
                "storage_type": "s3",
                "filename": filename,
                "bucket": self.s3_bucket,
                "download_url": download_url,
                "expires_in": self.s3_presigned_url_expiry,
                "available": True
            }
            
        except ClientError as e:
            if e.response['Error']['Code'] == '404':
                raise HTTPException(status_code=404, detail="File not found")
            else:
                logger.error(f"S3 error: {e}")
                raise HTTPException(status_code=500, detail=f"S3 error: {str(e)}")

    async def _get_local_download_info(self, filename: str) -> Dict[str, Any]:
        """
        Get local file download information.
        """
        file_path = os.path.join(self.local_storage_path, filename)
        
        if not os.path.exists(file_path):
            raise HTTPException(status_code=404, detail="File not found")
        
        return {
            "storage_type": "local",
            "filename": filename,
            "file_path": file_path,
            "download_url": f"/storage/download/{filename}",
            "file_size": os.path.getsize(file_path),
            "available": True
        }

    async def delete_file(self, filename: str) -> bool:
        """
        Delete a file from storage.
        """
        try:
            if self.storage_type == 's3':
                self.s3_client.delete_object(Bucket=self.s3_bucket, Key=filename)
                logger.info(f"Deleted file from S3: {filename}")
            else:
                file_path = os.path.join(self.local_storage_path, filename)
                if os.path.exists(file_path):
                    os.remove(file_path)
                    logger.info(f"Deleted file from local storage: {filename}")
                else:
                    raise HTTPException(status_code=404, detail="File not found")
            
            return True
            
        except ClientError as e:
            logger.error(f"S3 delete error: {e}")
            raise HTTPException(status_code=500, detail=f"S3 delete failed: {str(e)}")
        except Exception as e:
            logger.error(f"Delete error: {e}")
            raise HTTPException(status_code=500, detail=f"Delete failed: {str(e)}")

    async def list_files(self, prefix: str = "") -> Dict[str, Any]:
        """
        List files in storage with optional prefix filter.
        """
        try:
            if self.storage_type == 's3':
                response = self.s3_client.list_objects_v2(
                    Bucket=self.s3_bucket,
                    Prefix=prefix
                )
                
                files = []
                if 'Contents' in response:
                    for obj in response['Contents']:
                        files.append({
                            'filename': obj['Key'],
                            'size': obj['Size'],
                            'last_modified': obj['LastModified'].isoformat(),
                            'storage_url': self._generate_s3_presigned_url(obj['Key'])
                        })
                
                return {
                    "storage_type": "s3",
                    "bucket": self.s3_bucket,
                    "prefix": prefix,
                    "file_count": len(files),
                    "files": files
                }
            else:
                storage_path = self.local_storage_path
                if prefix:
                    storage_path = os.path.join(storage_path, prefix)
                
                files = []
                if os.path.exists(storage_path):
                    for filename in os.listdir(storage_path):
                        file_path = os.path.join(storage_path, filename)
                        if os.path.isfile(file_path):
                            files.append({
                                'filename': filename,
                                'size': os.path.getsize(file_path),
                                'last_modified': datetime.fromtimestamp(
                                    os.path.getmtime(file_path)
                                ).isoformat(),
                                'storage_url': f"/storage/download/{filename}"
                            })
                
                return {
                    "storage_type": "local",
                    "storage_path": storage_path,
                    "prefix": prefix,
                    "file_count": len(files),
                    "files": files
                }
                
        except Exception as e:
            logger.error(f"Error listing files: {e}")
            raise HTTPException(status_code=500, detail=f"Failed to list files: {str(e)}")

    async def get_storage_usage(self) -> Dict[str, Any]:
        """
        Get storage usage statistics.
        """
        try:
            if self.storage_type == 's3':
                # Note: This might be expensive for large buckets
                # Consider using AWS CloudWatch metrics instead
                response = self.s3_client.list_objects_v2(Bucket=self.s3_bucket)
                total_size = sum(obj['Size'] for obj in response.get('Contents', []))
                file_count = len(response.get('Contents', []))
                
                return {
                    "storage_type": "s3",
                    "bucket": self.s3_bucket,
                    "total_size_bytes": total_size,
                    "total_files": file_count,
                    "total_size_human": self._bytes_to_human(total_size)
                }
            else:
                total_size = 0
                file_count = 0
                
                for dirpath, dirnames, filenames in os.walk(self.local_storage_path):
                    for filename in filenames:
                        file_path = os.path.join(dirpath, filename)
                        total_size += os.path.getsize(file_path)
                        file_count += 1
                
                return {
                    "storage_type": "local",
                    "storage_path": self.local_storage_path,
                    "total_size_bytes": total_size,
                    "total_files": file_count,
                    "total_size_human": self._bytes_to_human(total_size)
                }
                
        except Exception as e:
            logger.error(f"Error getting storage usage: {e}")
            raise HTTPException(status_code=500, detail=f"Failed to get storage usage: {str(e)}")

    def _bytes_to_human(self, size_bytes: int) -> str:
        """
        Convert bytes to human readable format.
        """
        if size_bytes == 0:
            return "0B"
        
        size_names = ["B", "KB", "MB", "GB", "TB"]
        i = 0
        while size_bytes >= 1024 and i < len(size_names) - 1:
            size_bytes /= 1024.0
            i += 1
            
        return f"{size_bytes:.2f} {size_names[i]}"


# Create service instance
storage_service = StorageService()